"use client";

import { useEffect, useRef, useState } from "react";
import { motion, useInView } from "framer-motion";

interface CounterItem {
    img: string;
    value: number;
    suffix: string;
    text: string;
}

interface CounterSectionProps {
    data: CounterItem[];
}

export default function CounterSection({ data }: CounterSectionProps) {
    if (!data || data.length === 0) return null;

    return (
        <section className="counter-box py-5">
            <div className="container">
                <div className="row">
                    {data.map((item, index) => (
                        <CounterBox key={index} item={item} index={index} />
                    ))}
                </div>
            </div>
        </section>
    );
}

interface CounterBoxProps {
    item: CounterItem;
    index: number;
}

function CounterBox({ item, index }: CounterBoxProps) {
    const ref = useRef<HTMLDivElement | null>(null);
    const isInView = useInView(ref, { once: true });

    const [count, setCount] = useState<number>(0);

    useEffect(() => {
        if (!isInView) return;

        let start = 0;
        const end = item.value;
        const duration = 1500;
        const stepTime = 20;
        const increment = end / (duration / stepTime);

        const counter = setInterval(() => {
            start += increment;

            if (start >= end) {
                start = end;
                clearInterval(counter);
            }

            setCount(Math.floor(start));
        }, stepTime);

        return () => clearInterval(counter);
    }, [isInView, item.value]);

    return (
        <div className="col-lg-3 col-md-6" ref={ref}>
            <motion.div
                className="counter text-center p-3 shadow rounded"
                initial={{ opacity: 0, y: 40 }}
                whileInView={{ opacity: 1, y: 0 }}
                transition={{ delay: index * 0.2 }}
            >
                {item.img && (
                    <img
                        src={item.img}
                        width={60}
                        className="mb-3"
                        alt={item.text}
                    />
                )}

                <h3>
                    {count}
                    <span>{item.suffix}</span>
                </h3>

                <p>{item.text}</p>
            </motion.div>
        </div>
    );
}
