import React from 'react';
import { API_ENDPOINTS } from "@/lib/apiEndpoints";
import { Metadata } from "next";
import HandwrittenAssignmentsClient from "./HandwrittenAssignmentsClient";
import { buildAssignmentListMeta } from "@/lib/seo/assignmentMeta";

export const dynamic = "force-dynamic";

async function fetchHandwrittenAssignmentsData() {
    try {
        const [res, res1] = await Promise.all([
            API_ENDPOINTS.PAGE(4),
            API_ENDPOINTS.ASSIGNMENTS.CATEGORIES_COURSES('handwritten'),
        ]);

        if (!res.ok) throw new Error(`HTTP ${res.status}`);
        if (!res1.ok) throw new Error(`HTTP ${res1.status}`);

        const [pageData, categoriesData] = await Promise.all([
            res.json(),
            res1.json(),
        ]);

        return { pageData, categoriesData };
    } catch (err: any) {
        console.error("❌ Failed to load handwritten assignments data", err);
        return null;
    }
}

export async function generateMetadata(): Promise<Metadata> {
    const data = await fetchHandwrittenAssignmentsData();
    if (!data || !data.pageData) return { title: "Handwritten Assignments" };

    // Use buildAssignmentListMeta helper
    const meta = buildAssignmentListMeta(data.pageData, "Handwritten Assignments | IGNOU");
    return {
        ...meta,
        alternates: {
            canonical: "/assignments/handwritten",
        }
    };
}

export default async function HandwrittenAssignmentsPage() {
    const data = await fetchHandwrittenAssignmentsData();

    if (!data) {
        return <div className="text-danger text-center py-5">Failed to load handwritten assignments.</div>;
    }

    return <HandwrittenAssignmentsClient pageData={data.pageData} categoriesData={data.categoriesData} />;
}
