import { API_ENDPOINTS } from "@/lib/apiEndpoints";
import { Metadata } from "next";
import HardCopyAssignmentClient from "./HardCopyAssignmentClient";
import { buildAssignmentListMeta } from "@/lib/seo/assignmentMeta";

export const dynamic = "force-dynamic";

async function fetchHardCopyData() {
    try {
        const [resPage, resSection, resAssignments] = await Promise.all([
            API_ENDPOINTS.PAGE(11),
            API_ENDPOINTS.SECTION(93),
            API_ENDPOINTS.HANDWRITTEN.DATA(),
        ]);

        if (!resPage.ok) throw new Error(`PAGE HTTP ${resPage.status}`);
        if (!resSection.ok) throw new Error(`SECTION HTTP ${resSection.status}`);
        if (!resAssignments.ok) throw new Error(`ASSIGNMENTS HTTP ${resAssignments.status}`);

        const [pageData, sectionData, assignmentsData] = await Promise.all([
            resPage.json(),
            resSection.json(),
            resAssignments.json(),
        ]);

        return {
            pageData,
            sectionData: sectionData.description,
            categoriesTree: assignmentsData.categories_tree || [],
            allLanguages: assignmentsData.all_languages || [],
            allSessions: assignmentsData.all_sessions || [],
        };
    } catch (err: any) {
        console.error("❌ Failed to load hard copy data", err);
        return null;
    }
}

export async function generateMetadata(): Promise<Metadata> {
    const data = await fetchHardCopyData();
    if (!data || !data.pageData) return { title: "Hard Copy Assignments" };

    // Reuse list meta helper or create specific if needed
    const meta = buildAssignmentListMeta(data.pageData, "Hard Copy Assignments | IGNOU");
    return {
        ...meta,
        alternates: {
            canonical: "/assignments/hard-copy",
        }
    };
}

export default async function HardCopyPage() {
    const data = await fetchHardCopyData();

    if (!data) {
        return <div className="text-danger text-center py-5">Failed to load hard copy assignments.</div>;
    }

    return (
        <HardCopyAssignmentClient
            pageData={data.pageData}
            sectionDescription={data.sectionData}
            categoriesTree={data.categoriesTree}
            allLanguages={data.allLanguages}
            allSessions={data.allSessions}
        />
    );
}
