"use client";

import React, { useState } from 'react';
import InnerBanner from "../../../components/InnerBanner";
import "../page.css";
import Link from 'next/link';

function stripHtml(html: string): string {
    if (typeof window !== "undefined") {
        const div = document.createElement("div");
        div.innerHTML = html;
        return div.textContent || div.innerText || "";
    }
    return html.replace(/<[^>]*>?/gm, "").trim();
}

interface SolvedAssignmentsClientProps {
    pageData: any;
    categoriesData: any;
}

export default function SolvedAssignmentsClient({ pageData, categoriesData }: SolvedAssignmentsClientProps) {
    const [activeTab, setActiveTab] = useState<any>("all");
    const [search, setSearch] = useState("");

    const headline = pageData?.page_headline || "";
    const breadcrumbHeadline = pageData?.breadcrumb_headline || "";
    const description = pageData?.page_description ? stripHtml(pageData.page_description) : "";
    const allCategories = categoriesData?.categories || [];

    const categories = [
        { id: "all", label: "All Courses", icon: "fa-list" },
        ...allCategories
    ];

    const subCategories: Record<string, Array<{ code: string; title: string; course_slug: string }>> =
        allCategories.reduce((acc: any, cat: any) => {
            acc[cat.id] = cat.items || [];
            return acc;
        }, { all: [] });

    // Merge All Categories for "All Courses"
    subCategories.all = allCategories.flatMap((c: any) => c.items || []);

    // Apply Search Filter
    const filteredCourses = (
        activeTab === "all"
            ? subCategories.all
            : subCategories[activeTab] || []
    ).filter(
        (item: any) =>
            item.title.toLowerCase().includes(search.toLowerCase()) ||
            item.code.toLowerCase().includes(search.toLowerCase())
    );

    return (
        <>
            <InnerBanner title={headline} breadcrumb={breadcrumbHeadline} />

            <div className="projects-page">
                <div className="container">
                    {/* Heading */}
                    <h2>{headline}</h2>
                    <p dangerouslySetInnerHTML={{ __html: pageData?.page_description || "" }} />

                    <div className="d-lg-flex d-md-block d-block align-items-center gap-2 mb-4 justify-content-between">
                        <div className='d-flex'>
                            <i className="fa-solid fa-layer-group" style={{ fontSize: "1.75rem", color: "#292929ff" }}></i>
                            <h1 className="h3 ms-2 fw-bold text-dark">Browse by Category</h1>
                        </div>

                        {/* 🔥 Search Box */}
                        <div className="search-box mb-4 w-50 position-relative">
                            <input
                                type="text"
                                className="form-control "
                                placeholder="Search by Title or Course Code..."
                                value={search}
                                onChange={(e) => setSearch(e.target.value)}
                            />

                            <i className="fa-solid fa-magnifying-glass"></i>
                        </div>
                    </div>

                    {/* Tabs */}
                    <div className="projects-tabs mb-4">
                        {categories.map((cat) => (
                            <button
                                key={cat.id}
                                type="button"
                                onClick={() => setActiveTab(cat.id)}
                                className={`projects-tab-btn ${activeTab === cat.id ? "active" : ""}`}
                            >
                                <i className={`fa-solid ${cat.icon}`}></i>
                                {cat.label}
                            </button>
                        ))}
                    </div>

                    {/* Courses Grid */}
                    <div className="row g-4 mb-5">
                        {filteredCourses.map((item: any, i: number) => (
                            <div key={i} className="col-12 col-sm-6 col-md-6 col-lg-4 col-xl-3">
                                <Link href={`/assignments/solved/${item.course_slug}`} className='text-decoration-none'>
                                    <div className="course-card">
                                        <div className="course-card-inner">
                                            <div className="course-icon-badge">
                                                <i className="fa-solid fa-folder-open"></i>
                                            </div>
                                            <div className="coures-text-box">
                                                <h3 className="h6 mb-2 text-dark fw-semibold">{item.title}</h3>
                                                <span className="course-code-pill">{item.code}</span>
                                            </div>
                                        </div>
                                    </div>
                                </Link>
                            </div>
                        ))}
                    </div>

                </div>
            </div>
        </>
    );
}
