"use client";

import React, { useState, useMemo, useEffect } from "react";
import { AUTH, getToken } from "@/lib/apiEndpoints";
import Link from "next/link";
import InnerBanner from "../../components/InnerBanner";
import "./checkout.css";
import { useCart } from "@/context/CartContext";
import { useRouter } from "next/navigation";


export default function CheckoutPage() {
    const router = useRouter();
    const auth = AUTH();
    const isLoggedIn = !!getToken();

    useEffect(() => {
        if (!isLoggedIn) {
            router.replace("/login");
        }
    }, [isLoggedIn, router]);

    const { items, buyNowItem, isLoading, clearCart, clearBuyNowItem, } = useCart();
    const [orderCompleted, setOrderCompleted] = useState(false);

    const [billing, setBilling] = useState({
        firstName: '',
        lastName: '',
        email: '',
        phone: '',
        address: '',
        country: 'IN',
        state: '',
        zip: '',
    });

    const [shipping, setShipping] = useState({
        firstName: '',
        lastName: '',
        address: '',
        country: 'IN',
        state: '',
        zip: '',
    });

    const [paymentMethod, setPaymentMethod] = useState("card");
    const [sameAsBilling, setSameAsBilling] = useState(true);
    const [isPlacingOrder, setIsPlacingOrder] = useState(false);
    const [formError, setFormError] = useState<string | null>(null);

    const isBuyNow = !!buyNowItem;

    const checkoutItems = useMemo(() => {
        return buyNowItem ? [buyNowItem] : items;
    }, [isBuyNow, buyNowItem, items]);
    console.log(checkoutItems);

    const pageName = useMemo(() => {
        return isBuyNow
            ? buyNowItem?.pageName ?? null
            : "cart";
    }, [isBuyNow, buyNowItem]);

    // useEffect(() => {
    //     if (!isLoading && checkoutItems.length === 0 && !orderCompleted) {
    //         router.replace("/cart");
    //     }
    // }, [checkoutItems, isLoading, orderCompleted, router]);

    const subtotal = useMemo(
        () => checkoutItems.reduce((sum, item) => sum + item.price * item.qty, 0),
        [checkoutItems]
    );
    const total = subtotal; // Add shipping logic if needed

    /* -------------------- VALIDATION -------------------- */

    const REQUIRED_BILLING_FIELDS = [
        "firstName",
        "lastName",
        "email",
        "phone",
        "address",
        "state",
        "zip",
    ];

    const REQUIRED_SHIPPING_FIELDS = [
        "firstName",
        "lastName",
        "address",
        "state",
        "zip",
    ];

    const validateFields = (
        data: Record<string, string>,
        required: string[]
    ): string | null => {
        for (const field of required) {
            if (!data[field] || data[field].trim() === "") {
                return field;
            }
        }
        return null;
    };

    const handlePlaceOrder = async () => {
        if (!("PLACE_ORDER" in auth) || !auth.PLACE_ORDER || isPlacingOrder) {
            alert("Authentication error. Please login again.");
            router.push("/login");
            return;
        }

        if (isPlacingOrder) return;

        // Cart validation
        if (checkoutItems.length === 0) {
            alert("No items to checkout");
            return;
        }

        // Billing validation
        const missingBilling = validateFields(billing, REQUIRED_BILLING_FIELDS);
        if (missingBilling) {
            setFormError(missingBilling);
            // alert(`Please enter ${missingBilling}`);
            return;
        }

        // Shipping validation (only if different)
        if (!sameAsBilling) {
            const missingShipping = validateFields(shipping, REQUIRED_SHIPPING_FIELDS);
            if (missingShipping) {
                setFormError(`shipping-${missingShipping}`);
                // alert(`Please enter shipping ${missingShipping}`);
                return;
            }
        }

        setFormError(null);
        setIsPlacingOrder(true);

        const shippingDetails = sameAsBilling
            ? {
                firstName: billing.firstName,
                lastName: billing.lastName,
                address: billing.address,
                country: billing.country,
                state: billing.state,
                zip: billing.zip,
            }
            : shipping;

        const orderData = {
            billingDetails: billing,
            shippingDetails,
            paymentMethod,
            items: checkoutItems,
            subtotal,
            total,
            orderSource: isBuyNow ? "BUY_NOW" : "CART",
            pageName,
        };

        try {
            const response = await auth.PLACE_ORDER(orderData);
            const result = await response.json();

            if (!response.ok) {
                router.push(`/order/failed?message=${encodeURIComponent(result.message)}`);
                return;
            }

            setOrderCompleted(true);

            // ✅ CLEAR STATE CORRECTLY
            if (isBuyNow) {
                clearBuyNowItem();
            }
            else {
                clearCart();
            }

            if (result.success) {
                router.push(`/order/success?order=${result.order_number}`);
            } else {
                router.push(`/order/pending?order=${result.order_number}`);
            }

        } catch (error) {
            console.error('Error placing order:', error);
        } finally {
            setIsPlacingOrder(false);
        }
    };

    return (
        <>
            <InnerBanner title="Checkout" breadcrumb="Checkout" />

            <section className="checkout-section py-5">
                <div className="container">
                    <div className="row">
                        {/* Billing & Shipping */}
                        <div className="col-lg-8 mb-5 mb-lg-0">
                            <div className="checkout-form-card">
                                <h3 className="mb-4">Billing Details</h3>

                                <form>
                                    <fieldset disabled={isPlacingOrder}>
                                        {/* Billing Details */}
                                        <div className="row g-3">
                                            <div className="col-md-6">
                                                <label className="form-label">First Name</label>
                                                <input
                                                    type="text"
                                                    className={`form-control ${formError === "firstName" ? "is-invalid" : ""}`}
                                                    value={billing.firstName}
                                                    onChange={(e) =>
                                                        setBilling({ ...billing, firstName: e.target.value })
                                                    }
                                                />
                                                {formError === "firstName" && (
                                                    <div className="invalid-feedback">First Name is required</div>
                                                )}
                                            </div>

                                            <div className="col-md-6">
                                                <label className="form-label">Last Name</label>
                                                <input
                                                    type="text"
                                                    className={`form-control ${formError === "lastName" ? "is-invalid" : ""}`}
                                                    value={billing.lastName}
                                                    onChange={(e) =>
                                                        setBilling({ ...billing, lastName: e.target.value })
                                                    }
                                                />
                                                {formError === "lastName" && (
                                                    <div className="invalid-feedback">Last Name is required</div>
                                                )}
                                            </div>

                                            <div className="col-12">
                                                <label className="form-label">E-Mail Address</label>
                                                <input
                                                    type="email"
                                                    className={`form-control ${formError === "email" ? "is-invalid" : ""}`}
                                                    value={billing.email}
                                                    onChange={(e) =>
                                                        setBilling({ ...billing, email: e.target.value })
                                                    }
                                                />
                                                {formError === "email" && (
                                                    <div className="invalid-feedback">E-Mail is required</div>
                                                )}
                                            </div>

                                            <div className="col-12">
                                                <label className="form-label">Phone Number</label>
                                                <input
                                                    type="tel"
                                                    className={`form-control ${formError === "phone" ? "is-invalid" : ""}`}
                                                    value={billing.phone}
                                                    onChange={(e) =>
                                                        setBilling({ ...billing, phone: e.target.value })
                                                    }
                                                />
                                                {formError === "phone" && (
                                                    <div className="invalid-feedback">Phone is required</div>
                                                )}
                                            </div>

                                            <div className="col-12">
                                                <label className="form-label">Street Address</label>
                                                <input
                                                    type="text"
                                                    className={`form-control ${formError === "address" ? "is-invalid" : ""}`}
                                                    value={billing.address}
                                                    onChange={(e) =>
                                                        setBilling({ ...billing, address: e.target.value })
                                                    }
                                                />
                                                {formError === "address" && (
                                                    <div className="invalid-feedback">Address is required</div>
                                                )}
                                            </div>

                                            <div className="col-md-5">
                                                <label className="form-label">Country</label>
                                                <select
                                                    className="form-select"
                                                    value={billing.country}
                                                    onChange={(e) =>
                                                        setBilling({ ...billing, country: e.target.value })
                                                    }
                                                >
                                                    <option value="IN">India</option>
                                                    <option value="US">United States</option>
                                                </select>
                                            </div>

                                            <div className="col-md-4">
                                                <label className="form-label">State</label>
                                                <select
                                                    className={`form-select ${formError === "state" ? "is-invalid" : ""}`}
                                                    value={billing.state}
                                                    onChange={(e) =>
                                                        setBilling({ ...billing, state: e.target.value })
                                                    }
                                                >
                                                    <option value="">Select</option>
                                                    <option value="DL">Delhi</option>
                                                    <option value="MH">Maharashtra</option>
                                                    <option value="KA">Karnataka</option>
                                                </select>
                                                {formError === "state" && (
                                                    <div className="invalid-feedback">State is required</div>
                                                )}
                                            </div>

                                            <div className="col-md-3">
                                                <label className="form-label">Zip</label>
                                                <input
                                                    type="text"
                                                    className={`form-control ${formError === "zip" ? "is-invalid" : ""}`}
                                                    value={billing.zip}
                                                    onChange={(e) =>
                                                        setBilling({ ...billing, zip: e.target.value })
                                                    }
                                                />
                                                {formError === "zip" && (
                                                    <div className="invalid-feedback">Zip is required</div>
                                                )}
                                            </div>
                                        </div>

                                        {/* Same as Billing */}
                                        <div className="form-check mt-4">
                                            <input
                                                className="form-check-input"
                                                type="checkbox"
                                                id="sameAsBilling"
                                                checked={sameAsBilling}
                                                onChange={(e) => setSameAsBilling(e.target.checked)}
                                            />
                                            <label htmlFor="sameAsBilling" className="form-check-label fw-semibold">
                                                Shipping address same as Billing address
                                            </label>
                                        </div>

                                        {/* Shipping Details */}
                                        {!sameAsBilling && (
                                            <>
                                                <h4 className="mt-4 mb-3">Shipping Details</h4>
                                                <div className="row g-3">
                                                    <div className="col-md-6">
                                                        <label className="form-label">First Name</label>
                                                        <input
                                                            type="text"
                                                            className={`form-control ${formError === "shipping-firstName" ? "is-invalid" : ""}`}
                                                            value={shipping.firstName}
                                                            onChange={(e) =>
                                                                setShipping({ ...shipping, firstName: e.target.value })
                                                            }
                                                        />
                                                        {formError === "shipping-firstName" && (
                                                            <div className="invalid-feedback">First Name is required</div>
                                                        )}
                                                    </div>

                                                    <div className="col-md-6">
                                                        <label className="form-label">Last Name</label>
                                                        <input
                                                            type="text"
                                                            className={`form-control ${formError === "shipping-lastName" ? "is-invalid" : ""}`}
                                                            value={shipping.lastName}
                                                            onChange={(e) =>
                                                                setShipping({ ...shipping, lastName: e.target.value })
                                                            }
                                                        />
                                                        {formError === "shipping-lastName" && (
                                                            <div className="invalid-feedback">Last Name is required</div>
                                                        )}
                                                    </div>

                                                    <div className="col-12">
                                                        <label className="form-label">Street Address</label>
                                                        <input
                                                            type="text"
                                                            className={`form-control ${formError === "shipping-address" ? "is-invalid" : ""}`}
                                                            value={shipping.address}
                                                            onChange={(e) =>
                                                                setShipping({ ...shipping, address: e.target.value })
                                                            }
                                                        />
                                                        {formError === "shipping-address" && (
                                                            <div className="invalid-feedback">Address is required</div>
                                                        )}
                                                    </div>

                                                    <div className="col-md-5">
                                                        <label className="form-label">Country</label>
                                                        <select
                                                            className="form-select"
                                                            value={shipping.country}
                                                            onChange={(e) =>
                                                                setShipping({ ...shipping, country: e.target.value })
                                                            }
                                                        >
                                                            <option value="IN">India</option>
                                                            <option value="US">United States</option>
                                                        </select>
                                                    </div>

                                                    <div className="col-md-4">
                                                        <label className="form-label">State</label>
                                                        <select
                                                            className={`form-select ${formError === "shipping-state" ? "is-invalid" : ""}`}
                                                            value={shipping.state}
                                                            onChange={(e) =>
                                                                setShipping({ ...shipping, state: e.target.value })
                                                            }
                                                        >
                                                            <option value="">Select</option>
                                                            <option value="DL">Delhi</option>
                                                            <option value="MH">Maharashtra</option>
                                                            <option value="KA">Karnataka</option>
                                                        </select>
                                                        {formError === "shipping-state" && (
                                                            <div className="invalid-feedback">State is required</div>
                                                        )}
                                                    </div>

                                                    <div className="col-md-3">
                                                        <label className="form-label">Zip</label>
                                                        <input
                                                            type="text"
                                                            className={`form-control ${formError === "shipping-zip" ? "is-invalid" : ""}`}
                                                            value={shipping.zip}
                                                            onChange={(e) =>
                                                                setShipping({ ...shipping, zip: e.target.value })
                                                            }
                                                        />
                                                        {formError === "shipping-zip" && (
                                                            <div className="invalid-feedback">Zip is required</div>
                                                        )}
                                                    </div>
                                                </div>
                                            </>
                                        )}

                                        {/* Payment Method */}
                                        <div className="payment-methods mt-5">
                                            <h4 className="mb-3">Payment Method</h4>

                                            {["card", "upi", "cod"].map((method) => (
                                                <div
                                                    key={method}
                                                    className={`payment-option ${paymentMethod === method ? "selected" : ""}`}
                                                    onClick={() => setPaymentMethod(method)}
                                                >
                                                    <div className="form-check">
                                                        <input
                                                            className="form-check-input"
                                                            type="radio"
                                                            checked={paymentMethod === method}
                                                            readOnly
                                                        />
                                                        <label className="form-check-label fw-semibold text-capitalize">
                                                            {method === "cod"
                                                                ? "Cash on Delivery"
                                                                : method === "upi"
                                                                    ? "UPI / Net Banking"
                                                                    : "Credit / Debit Card"}
                                                        </label>
                                                    </div>
                                                </div>
                                            ))}
                                        </div>
                                    </fieldset>
                                </form>

                            </div>
                        </div>

                        {/* Order Summary */}
                        <div className="col-lg-4">
                            <div className="order-summary-card">
                                <div className="d-flex justify-content-between align-items-center mb-4">
                                    <h4 className="mb-0">Order Summary</h4>
                                    {!isBuyNow && (
                                        <Link href="/cart" className="text-decoration-none text-primary small">
                                            Edit Cart
                                        </Link>
                                    )}
                                </div>

                                <div className="cart-items mb-4">
                                    {isLoading ? (
                                        <div className="text-center py-4">
                                            <div
                                                className="spinner-border spinner-border-sm"
                                                role="status"
                                                style={{ color: "#2f80ff" }}
                                            >
                                                <span className="visually-hidden">Loading...</span>
                                            </div>
                                            <p className="text-muted small mt-2 mb-0">
                                                Loading summary...
                                            </p>
                                        </div>
                                    ) : checkoutItems.length === 0 ? (
                                        <p className="text-muted text-center mb-0">
                                            No items in cart
                                        </p>
                                    ) : (
                                        <>
                                            {checkoutItems.map((item) => (
                                                <div key={item.id} className="summary-item">
                                                    <div className="d-flex align-items-center">
                                                        {item.img && (
                                                            <img
                                                                src={item.img}
                                                                alt={item.name}
                                                                className="summary-img"
                                                            />
                                                        )}
                                                        <div className="summary-details">
                                                            <h6>{item.name}</h6>
                                                            <small>Qty: {item.qty}</small>
                                                        </div>
                                                    </div>
                                                    <span className="fw-semibold">
                                                        ₹{item.price * item.qty}
                                                    </span>
                                                </div>
                                            ))}

                                            <div className="d-flex justify-content-between mb-2">
                                                <span className="text-muted">Subtotal</span>
                                                <span className="fw-semibold">₹{subtotal}</span>
                                            </div>

                                            <div className="d-flex justify-content-between mb-2">
                                                <span className="text-muted">Shipping</span>
                                                <span className="text-success">Free</span>
                                            </div>

                                            <div className="total-row">
                                                <span>Total</span>
                                                <span>₹{total}</span>
                                            </div>

                                            <button
                                                type="button"
                                                className="btn-place-order mt-4"
                                                onClick={handlePlaceOrder}
                                                disabled={isPlacingOrder}
                                            >
                                                {isPlacingOrder ? (
                                                    <>
                                                        <span
                                                            className="spinner-border spinner-border-sm me-2"
                                                            role="status"
                                                            aria-hidden="true"
                                                        />
                                                        Placing Order...
                                                    </>
                                                ) : (
                                                    "Place Order"
                                                )}
                                            </button>

                                            <p className="text-center text-muted small mt-3">
                                                <i className="fa-solid fa-lock me-1"></i>
                                                Secure Checkout
                                            </p>
                                        </>
                                    )}
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </section>
        </>
    );
}
