"use client";

import React, { useState } from "react";
import InnerBanner from "../../components/InnerBanner";
import { Mail, Phone, MapPin, Facebook, Instagram, Linkedin, Youtube, Twitter, MessageCircle } from "lucide-react";
import "./contact.css";
import { API_ENDPOINTS } from "@/lib/apiEndpoints";

function stripHtml(html: string): string {
    if (typeof window !== "undefined") {
        const div = document.createElement("div");
        div.innerHTML = html;
        return div.textContent || div.innerText || "";
    }
    return html.replace(/<[^>]*>?/gm, "").trim();
}

const FAQItem = ({ question, answer }: { question: string; answer: string }) => {
    const [isOpen, setIsOpen] = useState(false);

    return (
        <div className={`faq-item ${isOpen ? "open" : ""}`} onClick={() => setIsOpen(!isOpen)}>
            <div className="faq-question">
                <span>{question}</span>
                <span className="faq-toggle">{isOpen ? "−" : "+"}</span>
            </div>
            <div className="faq-answer">
                <p>{answer}</p>
            </div>
        </div>
    );
};

interface CompanyContact {
    email: string;
    phone: string;
    alternate_phone: string;
    location: string;
    facebook_url: string;
    x_url: string;
    linkedin_url: string;
    youtube_url: string;
    instagram_url: string;
    telegram_url: string;
    whatsapp_url: string;
}

interface ContactClientProps {
    pageData: any;
    companyContact: CompanyContact | null;
    sectionData: any;
    faqsData: any[];
}

export default function ContactClient({ pageData, companyContact, sectionData, faqsData }: ContactClientProps) {
    const breadcrumbHeadline = pageData?.breadcrumb_headline || "";
    const headline = pageData?.page_headline || "";
    const description = pageData?.page_description ? stripHtml(pageData.page_description) : "";
    const image = pageData?.page_image || "";
    const sectionHeadline = sectionData?.section_headline || "";

    /* FORM STATES */
    const [formData, setFormData] = useState({
        first_name: "",
        last_name: "",
        email: "",
        phone: "",
        message: "",
    });

    const [loading, setLoading] = useState(false);
    const [error, setError] = useState("");
    const [success, setSuccess] = useState("");

    /* Supports input + textarea + select */
    const handleChange = (
        e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement | HTMLSelectElement>
    ) => {
        const { name, value } = e.target;

        let updatedValue = value;

        // Allow only digits for phone field
        if (name === "phone") {
            updatedValue = value
                .replace(/\D/g, "")
                .slice(0, 15);
        }

        setFormData({
            ...formData,
            [name]: updatedValue,
        });
    };

    /* SUBMIT FORM */
    const handleSubmit = async () => {
        setError("");
        setSuccess("");

        if (!formData.first_name || !formData.email || !formData.phone) {
            setError("Please fill all required fields.");
            return;
        }

        try {
            setLoading(true);

            const form = new FormData();
            form.append("first_name", formData.first_name);
            form.append("last_name", formData.last_name);
            form.append("email", formData.email);
            form.append("phone", formData.phone);
            form.append("message", formData.message);

            const res = await API_ENDPOINTS.CONTACT_ENQUIRY_SAVE(form);

            const data = await res.json();

            if (!res.ok) {
                throw new Error(data.message || "Something went wrong");
            }

            setSuccess("Message sent successfully!");
            setFormData({
                first_name: "",
                last_name: "",
                email: "",
                phone: "",
                message: "",
            });

        } catch (err: any) {
            setError(err.message || "Failed to send message.");
        } finally {
            setLoading(false);
        }
    };

    return (
        <>
            <InnerBanner title={breadcrumbHeadline} breadcrumb={breadcrumbHeadline} />

            <section className="contact-section py-5">
                <div className="container">
                    <div className="row">
                        {/* Left Column: Contact Info & Illustration */}
                        <div className="col-lg-6 mb-5 mb-lg-0">
                            <div className="contact-left-content">
                                <div className="section-header mb-4">
                                    <h2 className="display-5 fw-bold mb-3">{headline}</h2>
                                    <p className="text-muted" dangerouslySetInnerHTML={{ __html: pageData?.page_description || "" }} />
                                </div>

                                <div className="contact-illustration mb-5 text-center">
                                    <div style={{ minHeight: '400px', display: 'flex', alignItems: 'center', justifyContent: 'center' }}>
                                        <img src={image} alt={headline} className="img-fluid" style={{ maxHeight: '450px' }} onError={(e) => e.currentTarget.style.display = 'none'} />
                                    </div>
                                </div>

                                <div className="contact-details">
                                    <div className="d-flex align-items-start mb-4">
                                        <div className="icon-box me-3">
                                            <Mail size={24} className="text-primary" />
                                        </div>
                                        <div>
                                            <h6 className="fw-bold mb-1">E-Mail Id</h6>
                                            <a href={`mailto: ${companyContact?.email}`} className="text-decoration-none text-muted">{companyContact?.email}</a>
                                        </div>
                                    </div>

                                    <div className="d-flex align-items-start mb-4">
                                        <div className="icon-box me-3">
                                            <Phone size={24} className="text-primary" />
                                        </div>
                                        <div>
                                            <h6 className="fw-bold mb-1">Phone</h6>
                                            <a href={`tel: ${companyContact?.phone}`} className="text-decoration-none text-muted">{companyContact?.phone}</a>
                                        </div>
                                    </div>

                                    <div className="d-flex align-items-start mb-4">
                                        <div className="icon-box me-3">
                                            <MapPin size={24} className="text-primary" />
                                        </div>
                                        <div>
                                            <h6 className="fw-bold mb-1">Address</h6>
                                            <p className="text-muted mb-0">{stripHtml(companyContact?.location ?? "")}</p>
                                        </div>
                                    </div>
                                </div>

                                <div className="social-links mt-5">
                                    <h5 className="fw-bold mb-3">Follow us</h5>
                                    <div className="d-flex gap-3">
                                        {companyContact?.whatsapp_url && (
                                            <a href={companyContact.whatsapp_url} className="social-icon">
                                                <MessageCircle size={20} />
                                            </a>
                                        )}
                                        {companyContact?.linkedin_url && (
                                            <a href={companyContact.linkedin_url} className="social-icon">
                                                <Linkedin size={20} />
                                            </a>
                                        )}
                                        {companyContact?.x_url && (
                                            <a href={companyContact.x_url} className="social-icon">
                                                <Twitter size={20} />
                                            </a>
                                        )}
                                        {companyContact?.instagram_url && (
                                            <a href={companyContact.instagram_url} className="social-icon">
                                                <Instagram size={20} />
                                            </a>
                                        )}
                                        {companyContact?.facebook_url && (
                                            <a href={companyContact.facebook_url} className="social-icon">
                                                <Facebook size={20} />
                                            </a>
                                        )}
                                        {companyContact?.youtube_url && (
                                            <a href={companyContact.youtube_url} className="social-icon">
                                                <Youtube size={20} />
                                            </a>
                                        )}
                                    </div>
                                </div>
                            </div>
                        </div>

                        {/* Right Column: form */}
                        <div className="col-lg-6">
                            <div className="contact-form-wrapper p-lg-5 p-3 p-md-4">
                                <h2 className="display-5 fw-bold mb-5">Get in Touch</h2>

                                {error && (
                                    <div className="alert alert-danger py-2">
                                        <i className="fa-solid fa-circle-exclamation me-2"></i>{error}
                                    </div>
                                )}

                                {success && (
                                    <div className="alert alert-success py-2">
                                        <i className="fa-solid fa-circle-check me-2"></i>{success}
                                    </div>
                                )}

                                <form>
                                    <div className="row g-3">
                                        <div className="col-md-6">
                                            <label htmlFor="first_name" className="form-label">First Name</label>
                                            <input
                                                type="text"
                                                name="first_name"
                                                id="first_name"
                                                className="form-control"
                                                value={formData.first_name}
                                                onChange={handleChange}
                                                placeholder="Enter your First Name"
                                            />
                                        </div>
                                        <div className="col-md-6">
                                            <label htmlFor="last_name" className="form-label">Last Name</label>
                                            <input
                                                type="text"
                                                name="last_name"
                                                id="last_name"
                                                className="form-control"
                                                value={formData.last_name}
                                                onChange={handleChange}
                                                placeholder="Enter your Last Name"
                                            />
                                        </div>
                                        <div className="col-12">
                                            <label htmlFor="email" className="form-label">E-Mail</label>
                                            <input
                                                type="email"
                                                name="email"
                                                id="email"
                                                className="form-control"
                                                value={formData.email}
                                                onChange={handleChange}
                                                placeholder="Enter your E-Mail"
                                            />
                                        </div>
                                        <div className="col-12">
                                            <label htmlFor="phone" className="form-label">Mobile No.</label>
                                            <input
                                                type="tel"
                                                name="phone"
                                                id="phone"
                                                className="form-control"
                                                value={formData.phone}
                                                onChange={handleChange}
                                                placeholder="Enter your Phone"
                                            />
                                        </div>
                                        <div className="col-12">
                                            <label htmlFor="message" className="form-label">Message</label>
                                            <textarea
                                                name="message"
                                                id="message"
                                                className="form-control"
                                                rows={5}
                                                value={formData.message}
                                                onChange={handleChange}
                                                placeholder="Write your Message..."
                                            ></textarea>
                                        </div>

                                        {/* <div className="col-12">
                                            <label className="form-label d-block mb-3">Services</label>
                                            <div className="bg-white p-3 border rounded mb-3">
                                                <input type="text" className="form-control border-0" placeholder="Handwritten Hard Copy" readOnly />
                                            </div>

                                            <div className="d-flex flex-wrap gap-4 services-audit">
                                                <div className="form-check">
                                                    <input className="form-check-input" type="checkbox" value="" id="service1" />
                                                    <label className="form-check-label text-muted small" htmlFor="service1">
                                                        Solved Assignment
                                                    </label>
                                                </div>
                                                <div className="form-check">
                                                    <input className="form-check-input" type="checkbox" value="" id="service2" defaultChecked />
                                                    <label className="form-check-label text-muted small" htmlFor="service2">
                                                        Handwritten Hard Copy
                                                    </label>
                                                </div>
                                                <div className="form-check">
                                                    <input className="form-check-input" type="checkbox" value="" id="service3" />
                                                    <label className="form-check-label text-muted small" htmlFor="service3">
                                                        Project Report
                                                    </label>
                                                </div>
                                                <div className="form-check">
                                                    <input className="form-check-input" type="checkbox" value="" id="service4" />
                                                    <label className="form-check-label text-muted small" htmlFor="service4">
                                                        Others
                                                    </label>
                                                </div>
                                            </div>
                                        </div> */}

                                        <div className="col-12 mt-4">
                                            <button
                                                className="btn btn-primary px-4 py-2"
                                                onClick={handleSubmit}
                                                disabled={loading}
                                            >
                                                {loading ? "Sending..." : "Send Message"}
                                            </button>
                                        </div>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>
            </section>

            {/* FAQ Section */}
            <section className="faq-section py-5 bg-white">
                <div className="container">
                    <div className="row align-items-center mb-5">
                        <div className="col-lg-4">
                            <h2 className="display-4 fw-bold serif-font text-primary fst-italic">{sectionHeadline}</h2>
                        </div>
                        <div className="col-lg-8">
                            <div className="faq-list">
                                {faqsData && faqsData.map((faq: any, index: number) => (
                                    <FAQItem
                                        key={index}
                                        question={faq.question}
                                        answer={stripHtml(faq.answer)}
                                    />
                                ))}
                            </div>
                        </div>
                    </div>
                </div>
            </section>
        </>
    );
}
