import React from "react";
import { API_ENDPOINTS } from "@/lib/apiEndpoints";
import { Metadata } from "next";
import ContactClient from "./ContactClient";

export const dynamic = "force-dynamic";

function stripHtml(html: string): string {
    if (!html) return "";
    return html.replace(/<[^>]*>?/gm, "").trim();
}

async function fetchContactData() {
    try {
        const [res, res2, res3, faqs_res] = await Promise.all([
            API_ENDPOINTS.PAGE(7),
            API_ENDPOINTS.COMPANY_CONTACT(),
            API_ENDPOINTS.SECTION(50),
            API_ENDPOINTS.FAQS(),
        ]);

        if (!res.ok) throw new Error(`HTTP ${res.status}`);
        if (!res2.ok) throw new Error(`HTTP ${res2.status}`);
        if (!res3.ok) throw new Error(`HTTP ${res3.status}`);
        if (!faqs_res.ok) throw new Error(`HTTP ${faqs_res.status}`);

        const [pageData, companyContact, sectionData, faqsData] = await Promise.all([
            res.json(),
            res2.json(),
            res3.json(),
            faqs_res.json(),
        ]);

        return {
            pageData,
            companyContact,
            sectionData,
            faqsData
        };
    } catch (err: any) {
        console.error("❌ Failed to load contact data", err);
        return null;
    }
}

import { buildPageMeta } from "@/lib/seo/generalMeta";

export async function generateMetadata(): Promise<Metadata> {
    const data = await fetchContactData();

    if (!data || !data.pageData) {
        return {
            title: "Contact Us",
            description: "Get in touch with us for any queries."
        };
    }

    return buildPageMeta(data.pageData, "/contact", "Contact Us");
}

export default async function ContactPage() {
    const data = await fetchContactData();

    if (!data) {
        return <div className="text-danger text-center py-5">Failed to load contact page.</div>;
    }

    return (
        <ContactClient
            pageData={data.pageData}
            companyContact={data.companyContact}
            sectionData={data.sectionData}
            faqsData={data.faqsData}
        />
    );
}