"use client";

import React, { useEffect, useState, useRef, useMemo } from 'react';
import { AUTH, getToken } from "@/lib/apiEndpoints";
import { useRouter } from "next/navigation";
import { useCart } from '@/context/CartContext';

// Type Definitions
interface TrackingStep {
    step: string;
    date: string;
    completed: boolean;
}

interface Order {
    id: string;
    date: string;
    total: number;
    status: string;
    statusColor: string;
    items: number;
    trackingSteps: TrackingStep[];
}

// interface CartItem {
//     id: number;
//     name: string;
//     price: number;
//     quantity: number;
//     image: string;
// }

interface Address {
    id: number;
    type: string;
    address: string;
    isDefault: boolean;
}

// interface Card {
//     id: number;
//     type: string;
//     last4: string;
//     expiry: string;
//     isDefault: boolean;
// }

type UserProfile = {
    first_name: string;
    last_name: string;
    email: string;
    phone: string;
    profile_photo?: string;
};

type CustomerProfileResponse = {
    status: "success" | "error";
    message?: string;
    customer: UserProfile;
};

interface ProfileComponentProps {
    userProfile: UserProfile | null;
    setUserProfile: (profile: UserProfile) => void;
    loadingProfile: boolean;
}

interface CardForm {
    holder: string;
    type: string;
    number: string;
    month: string;
    year: string;
    cvv: string;
    isDefault: boolean;
}

interface Order {
    order_number: string;
    total: number;
    order_status: string;
    created_at: string;
}

interface OrdersComponentProps {
    orders: Order[];
    setOrders: React.Dispatch<React.SetStateAction<Order[]>>;
    loadingOrders: boolean;
}

interface CustomerCard {
    encrypted_id: string;
    holder_name: string;
    card_type: string;
    last4: string;
    expiry_month: string;
    expiry_year: string;
    is_default: boolean;
}

interface CardForm {
    holder: string;
    type: string;
    number: string;
    month: string;
    year: string;
    cvv: string;
    isDefault: boolean;
}

interface CardsComponentProps {
    cards: CustomerCard[];
    setCards: React.Dispatch<React.SetStateAction<CustomerCard[]>>;
    loadingCards: boolean;
}

interface PasswordForm {
    currentPassword: string;
    newPassword: string;
    confirmPassword: string;
}

interface PasswordErrors {
    currentPassword?: string;
    newPassword?: string;
    confirmPassword?: string;
}

// Main Dashboard Component
export default function UserDashboard() {
    const router = useRouter();
    const auth = AUTH();
    const isLoggedIn = !!getToken();

    const [checking, setChecking] = useState(true);
    useEffect(() => {
        const token = getToken();

        if (!token) {
            router.replace("/login");
        } else {
            setChecking(false);
        }
    }, [router]);

    const [activeTab, setActiveTab] = useState('dashboard');
    const [isAuthenticated, setIsAuthenticated] = useState(true);
    const [authView, setAuthView] = useState('login');
    const [isSidebarOpen, setIsSidebarOpen] = useState(true);

    const [selectedOrder, setSelectedOrder] = useState<Order | null>(null);
    const [showAddCard, setShowAddCard] = useState(false);
    const [showAddAddress, setShowAddAddress] = useState(false);

    const { items: cartItems } = useCart();

    const [addresses, setAddresses] = useState<Address[]>([
        { id: 1, type: 'Home', address: '123 Main St, Mumbai, MH 400001', isDefault: true },
        { id: 2, type: 'Office', address: '456 Park Ave, Delhi, DL 110001', isDefault: false }
    ]);

    // const [cards, setCards] = useState<Card[]>([
    //     { id: 1, type: 'Visa', last4: '4242', expiry: '12/25', isDefault: true },
    //     { id: 2, type: 'Mastercard', last4: '5555', expiry: '06/26', isDefault: false }
    // ]);

    // Authentication Component
    const AuthComponent = () => {
        if (authView === 'login') {
            return (
                <div className="card dashboard-card">
                    <div className="card-body p-4">
                        <h3 className="text-center mb-4"><i className="fas fa-user-circle me-2" style={{ color: '#2f80ff' }}></i>Login</h3>
                        <div>
                            <div className="mb-3">
                                <label className="form-label"><i className="fas fa-envelope me-2"></i>Email</label>
                                <input type="email" className="form-control" placeholder="Enter your email" />
                            </div>
                            <div className="mb-3">
                                <label className="form-label"><i className="fas fa-lock me-2"></i>Password</label>
                                <input type="password" className="form-control" placeholder="Enter password" />
                            </div>
                            <button type="button" className="btn w-100 mb-3" style={{ backgroundColor: '#2f80ff', color: 'white' }} onClick={() => setIsAuthenticated(true)}>
                                <i className="fas fa-sign-in-alt me-2"></i>Login
                            </button>
                            <div className="text-center">
                                <a href="#" className="text-decoration-none me-3" style={{ color: '#2f80ff' }} onClick={(e) => { e.preventDefault(); setAuthView('forgot'); }}>Forgot Password?</a>
                                <a href="#" className="text-decoration-none" style={{ color: '#2f80ff' }} onClick={(e) => { e.preventDefault(); setAuthView('signup'); }}>Create Account</a>
                            </div>
                        </div>
                    </div>
                </div>
            );
        } else if (authView === 'signup') {
            return (
                <div className="card dashboard-card">
                    <div className="card-body p-4">
                        <h3 className="text-center mb-4"><i className="fas fa-user-plus me-2" style={{ color: '#2f80ff' }}></i>Create Account</h3>
                        <div>
                            <div className="mb-3">
                                <label className="form-label"><i className="fas fa-user me-2"></i>Full Name</label>
                                <input type="text" className="form-control" placeholder="Enter your name" />
                            </div>
                            <div className="mb-3">
                                <label className="form-label"><i className="fas fa-envelope me-2"></i>Email</label>
                                <input type="email" className="form-control" placeholder="Enter your email" />
                            </div>
                            <div className="mb-3">
                                <label className="form-label"><i className="fas fa-lock me-2"></i>Password</label>
                                <input type="password" className="form-control" placeholder="Create password" />
                            </div>
                            <div className="mb-3">
                                <label className="form-label"><i className="fas fa-lock me-2"></i>Confirm Password</label>
                                <input type="password" className="form-control" placeholder="Confirm password" />
                            </div>
                            <button type="button" className="btn w-100 mb-3" style={{ backgroundColor: '#2f80ff', color: 'white' }} onClick={() => setIsAuthenticated(true)}>
                                <i className="fas fa-user-check me-2"></i>Sign Up
                            </button>
                            <div className="text-center">
                                <a href="#" className="text-decoration-none" style={{ color: '#2f80ff' }} onClick={(e) => { e.preventDefault(); setAuthView('login'); }}>Already have an account? Login</a>
                            </div>
                        </div>
                    </div>
                </div>
            );
        } else {
            return (
                <div className="card dashboard-card">
                    <div className="card-body p-4">
                        <h3 className="text-center mb-4"><i className="fas fa-key me-2" style={{ color: '#2f80ff' }}></i>Forgot Password</h3>
                        <div>
                            <div className="mb-3">
                                <label className="form-label"><i className="fas fa-envelope me-2"></i>Email</label>
                                <input type="email" className="form-control" placeholder="Enter your email" />
                            </div>
                            <button type="button" className="btn w-100 mb-3" style={{ backgroundColor: '#2f80ff', color: 'white' }}>
                                <i className="fas fa-paper-plane me-2"></i>Send Reset Link
                            </button>
                            <div className="text-center">
                                <a href="#" className="text-decoration-none" style={{ color: '#2f80ff' }} onClick={(e) => { e.preventDefault(); setAuthView('login'); }}>Back to Login</a>
                            </div>
                        </div>
                    </div>
                </div>
            );
        }
    };

    // Profile Component
    const [userProfile, setUserProfile] = useState<UserProfile | null>(null);
    const [loadingProfile, setLoadingProfile] = useState(true);

    useEffect(() => {
        async function loadProfile() {
            try {
                const response = await auth.CUSTOMER_PROFILE();
                let data: CustomerProfileResponse;

                if (response instanceof Response) {
                    if (!response.ok) throw new Error("Failed to fetch profile");
                    data = (await response.json()) as CustomerProfileResponse;
                } else {
                    data = response as CustomerProfileResponse;
                }

                if (data.status !== "success") throw new Error(data.message || "Failed to fetch profile");

                setUserProfile(data.customer);
            } catch (err) {
                console.error("Failed to load profile", err);
            } finally {
                setLoadingProfile(false);
            }
        }

        loadProfile();
    }, []);

    const ProfileComponent: React.FC<ProfileComponentProps> = ({ userProfile, setUserProfile, loadingProfile }) => {
        const [firstName, setFirstName] = useState("");
        const [lastName, setLastName] = useState("");
        const [email, setEmail] = useState("");
        const [phone, setPhone] = useState<string>("");

        const [photoFile, setPhotoFile] = useState<File | null>(null);
        const [photoPreview, setPhotoPreview] = useState<string | null>(null);

        const fileInputRef = useRef<HTMLInputElement | null>(null);

        const [errors, setErrors] = useState<Record<string, string>>({});
        const [submitError, setSubmitError] = useState<string | null>(null);
        const [successMsg, setSuccessMsg] = useState<string | null>(null);
        const [isSubmitting, setIsSubmitting] = useState(false);

        // Initialize form fields when userProfile changes
        useEffect(() => {
            if (userProfile) {
                setFirstName(userProfile.first_name || "");
                setLastName(userProfile.last_name || "");
                setEmail(userProfile.email || "");
                setPhone(userProfile.phone || "");
                setPhotoPreview(userProfile.profile_photo || null);
            }
        }, [userProfile]);

        // ---------- Showing Message with Timeout ------------
        useEffect(() => {
            if (successMsg || submitError) {
                const timer = setTimeout(() => {
                    setSuccessMsg(null);
                    setSubmitError(null);
                }, 3000);
                return () => clearTimeout(timer);
            }
        }, [successMsg, submitError]);

        // ---------- Validation ----------
        const validate = () => {
            const newErrors: Record<string, string> = {};

            if (!firstName.trim()) newErrors.firstName = "First name is required.";
            // if (!lastName.trim()) newErrors.lastName = "Last name is required.";
            if (!email.trim()) newErrors.email = "E-Mail is required.";
            else if (!/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email)) newErrors.email = "Please enter a valid E-Mail address.";
            if (!phone.trim()) newErrors.phone = "Phone number is required.";
            else if (!/^[0-9]{8,15}$/.test(phone)) newErrors.phone = "Invalid phone number.";

            if (photoFile) {
                const allowedTypes = ["image/jpeg", "image/png", "image/jpg", "image/webp"];
                if (!allowedTypes.includes(photoFile.type)) newErrors.photo = "Only JPG, PNG, or WEBP images are allowed.";
                else if (photoFile.size > 2 * 1024 * 1024) newErrors.photo = "Image size must be less than 2MB.";
            }

            setErrors(newErrors);
            return Object.keys(newErrors).length === 0;
        };

        // ---------- Handlers ----------
        const handlePhotoClick = () => fileInputRef.current?.click();

        const handlePhotoChange = (e: React.ChangeEvent<HTMLInputElement>) => {
            const file = e.target.files?.[0];
            if (!file) return;

            const allowedTypes = ["image/jpeg", "image/png", "image/jpg", "image/webp"];
            if (!allowedTypes.includes(file.type)) {
                setErrors((prev) => ({ ...prev, photo: "Only JPG, PNG, or WEBP images are allowed." }));
                return;
            }
            if (file.size > 2 * 1024 * 1024) {
                setErrors((prev) => ({ ...prev, photo: "Image size must be less than 2MB." }));
                return;
            }

            setErrors((prev) => ({ ...prev, photo: "" }));
            setPhotoFile(file);
            setPhotoPreview(URL.createObjectURL(file));
        };

        const handleSubmit = async () => {
            if (!validate()) return;
            setSubmitError(null);
            setSuccessMsg(null);
            setIsSubmitting(true);

            try {
                const payload = new FormData();
                payload.append("first_name", firstName);
                payload.append("last_name", lastName ?? null);
                payload.append("email", email);
                payload.append("phone", phone);
                if (photoFile) payload.append("profile_photo", photoFile);

                const res = await auth.UPDATE_CUSTOMER_PROFILE(payload);
                let data: any;

                if (res instanceof Response) {
                    if (!res.ok) {
                        const err = await res.json();
                        throw new Error(err.message || "Update failed");
                    }
                    data = await res.json();
                } else {
                    data = res;
                    if (data.status !== "success") throw new Error(data.message || "Update failed");
                }

                // Use updated profile photo from server, if returned
                const updatedProfile = {
                    ...userProfile!,
                    first_name: firstName,
                    last_name: lastName,
                    email,
                    phone,
                    profile_photo: data.customer?.profile_photo || photoPreview,
                };

                setTimeout(() => {
                    setUserProfile(updatedProfile);
                    setPhotoPreview(updatedProfile.profile_photo);
                }, 1000);

                setSuccessMsg("Profile updated successfully!");
            } catch (err: any) {
                console.error("❌ Profile update failed:", err);
                setSubmitError(err.message || "Failed to update profile.");
            } finally {
                setIsSubmitting(false);
            }
        };

        if (loadingProfile) {
            return (
                <div className="card-body text-center">
                    <div className="spinner-border text-primary" />
                </div>
            );
        }

        if (!userProfile) {
            return <div className="card-body text-center text-muted">Profile data not available</div>;
        }

        return (
            <div className="card dashboard-card">
                <div className="card-body">
                    <h4 className="mb-4">
                        <i className="fas fa-user me-2" style={{ color: "#2f80ff" }}></i>
                        Profile Information
                    </h4>

                    <div className="text-start mb-4 d-block">
                        <img
                            src={photoPreview!}
                            className="rounded-circle mb-3"
                            alt="Profile"
                            style={{ border: "3px solid #2f80ff", width: 120, height: 120, objectFit: "cover" }}
                        />
                        <input
                            type="file"
                            ref={fileInputRef}
                            className="d-none"
                            accept="image/*"
                            onChange={handlePhotoChange}
                        />
                        <button
                            type="button"
                            className="btn btn-sm d-block mt-3"
                            style={{ backgroundColor: "#2f80ff", color: "white" }}
                            onClick={handlePhotoClick}
                        >
                            <i className="fas fa-camera me-2"></i> Change Photo
                        </button>
                        {errors.photo && <small className="text-danger d-block mt-2">{errors.photo}</small>}
                    </div>

                    <div>
                        <div className="row mb-3">
                            <div className="col-md-6">
                                <label className="form-label">First Name</label>
                                <input
                                    type="text"
                                    className={`form-control ${errors.firstName ? "is-invalid" : ""}`}
                                    value={firstName}
                                    onChange={(e) => setFirstName(e.target.value)}
                                />
                                {errors.firstName && <div className="invalid-feedback">{errors.firstName}</div>}
                            </div>
                            <div className="col-md-6">
                                <label className="form-label">Last Name</label>
                                <input
                                    type="text"
                                    className={`form-control ${errors.lastName ? "is-invalid" : ""}`}
                                    value={lastName}
                                    onChange={(e) => setLastName(e.target.value)}
                                />
                                {errors.lastName && <div className="invalid-feedback">{errors.lastName}</div>}
                            </div>
                        </div>

                        <div className="mb-3">
                            <label className="form-label">E-Mail</label>
                            <input
                                type="email"
                                className={`form-control ${errors.email ? "is-invalid" : ""}`}
                                value={email}
                                onChange={(e) => setEmail(e.target.value)}
                            />
                            {errors.email && <div className="invalid-feedback">{errors.email}</div>}
                        </div>

                        <div className="mb-3">
                            <label className="form-label">Phone</label>
                            <input
                                type="number"
                                className={`form-control ${errors.phone ? "is-invalid" : ""}`}
                                value={phone}
                                onChange={(e) => setPhone(e.target.value)}
                            />
                            {errors.phone && <div className="invalid-feedback">{errors.phone}</div>}
                        </div>

                        {submitError && <div className="alert alert-danger">{submitError}</div>}
                        {successMsg && <div className="alert alert-success">{successMsg}</div>}

                        <button
                            type="button"
                            className="btn"
                            style={{ backgroundColor: "#2f80ff", color: "white", minWidth: 160 }}
                            onClick={handleSubmit}
                            disabled={isSubmitting}
                        >
                            {isSubmitting ? (
                                <>
                                    <span className="spinner-border spinner-border-sm me-2" />
                                    Updating...
                                </>
                            ) : (
                                <>
                                    <i className="fas fa-save me-2"></i> Update Profile
                                </>
                            )}
                        </button>
                    </div>
                </div>
            </div>
        );
    };

    // Cart Component
    const CartComponent = () => {
        const { items: cartItems, isLoading, changeQty, removeFromCart } = useCart();

        const total = useMemo(
            () =>
                cartItems.reduce(
                    (sum, item) => sum + item.price * item.qty,
                    0
                ),
            [cartItems]
        );

        return (
            <div className="card dashboard-card">
                <div className="card-body">
                    <h4 className="mb-4">
                        <i
                            className="fas fa-shopping-cart me-2"
                            style={{ color: "#2f80ff" }}
                        ></i>
                        Shopping Cart ({cartItems.length} items)
                    </h4>

                    {isLoading ? (
                        <div className="text-center py-4">
                            <div
                                className="spinner-border"
                                role="status"
                                style={{ color: "#2f80ff" }}
                            >
                                <span className="visually-hidden">Loading...</span>
                            </div>
                            <p className="mt-3 text-muted">Loading your cart...</p>
                        </div>
                    ) : cartItems.length === 0 ? (
                        <div className="text-center py-4">
                            <i
                                className="fa-solid fa-cart-shopping fa-2x"
                                style={{ color: "#2f80ff" }}
                            ></i>
                            <p className="mt-3 text-muted">Your cart is empty.</p>
                        </div>
                    ) : (
                        <>
                            <div className="row">
                                {cartItems.map((item) => (
                                    <div key={item.id} className="col-lg-6">
                                        <div
                                            className="card mb-3 h-100"
                                            style={{ borderColor: "#2f80ff" }}
                                        >
                                            <div className="card-body">
                                                <div className="row align-items-center">
                                                    <div className="col-md-2">
                                                        <img
                                                            src={item.img}
                                                            className="img-fluid rounded"
                                                            alt={item.name}
                                                        />
                                                    </div>

                                                    <div className="col-md-4">
                                                        <h6>{item.name}</h6>
                                                        <p className="text-muted mb-0">
                                                            ₹{item.price}
                                                        </p>
                                                    </div>

                                                    <div className="col-md-3">
                                                        <div className="input-group input-group-sm">
                                                            <button
                                                                className="btn"
                                                                style={{
                                                                    borderColor:
                                                                        "#2f80ff",
                                                                    color:
                                                                        "#2f80ff",
                                                                }}
                                                                onClick={() =>
                                                                    changeQty(
                                                                        item.id,
                                                                        -1
                                                                    )
                                                                }
                                                            >
                                                                <i className="fas fa-minus"></i>
                                                            </button>

                                                            <input
                                                                type="text"
                                                                className="form-control text-center"
                                                                value={item.qty}
                                                                readOnly
                                                            />

                                                            <button
                                                                className="btn"
                                                                style={{
                                                                    borderColor:
                                                                        "#2f80ff",
                                                                    color:
                                                                        "#2f80ff",
                                                                }}
                                                                onClick={() =>
                                                                    changeQty(
                                                                        item.id,
                                                                        1
                                                                    )
                                                                }
                                                            >
                                                                <i className="fas fa-plus"></i>
                                                            </button>
                                                        </div>
                                                    </div>

                                                    <div className="col-md-2">
                                                        <strong
                                                            style={{
                                                                color: "#2f80ff",
                                                            }}
                                                        >
                                                            ₹
                                                            {item.price *
                                                                item.qty}
                                                        </strong>
                                                    </div>

                                                    <div className="col-md-1">
                                                        <button
                                                            className="btn btn-sm btn-danger"
                                                            onClick={() =>
                                                                removeFromCart(
                                                                    item.id
                                                                )
                                                            }
                                                        >
                                                            <i className="fas fa-trash"></i>
                                                        </button>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                ))}
                            </div>

                            <div className="text-end">
                                <h5>
                                    Total:{" "}
                                    <span
                                        style={{
                                            color: "#2f80ff",
                                        }}
                                    >
                                        ₹{total}
                                    </span>
                                </h5>

                                {isLoggedIn ? (
                                    <a
                                        href="/checkout"
                                        className="btn btn-lg mt-3"
                                        style={{
                                            backgroundColor: "#2f80ff",
                                            color: "white",
                                        }}
                                    >
                                        <i className="fas fa-credit-card me-2"></i>
                                        Proceed to Checkout
                                    </a>
                                ) : (
                                    <a
                                        href="/login"
                                        className="btn btn-lg mt-3"
                                        style={{
                                            backgroundColor: "#2f80ff",
                                            color: "white",
                                        }}
                                    >
                                        <i className="fas fa-credit-card me-2"></i>
                                        Proceed to Checkout
                                    </a>
                                )}
                            </div>
                        </>
                    )}
                </div>
            </div>
        );
    };

    // Payment Cards Component
    const [cards, setCards] = useState<CustomerCard[]>([]);
    const [loadingCards, setLoadingCards] = useState(true);

    useEffect(() => {
        const loadCards = async () => {
            try {
                const response = await auth.CUSTOMER_CARDS.LIST();
                if (!response.ok) throw new Error();
                const data = await response.json();
                setCards(data?.data || []);
            } catch (err) {
                console.error("Failed to load cards", err);
            } finally {
                setLoadingCards(false);
            }
        };

        loadCards();
    }, []);
    const CardsComponent: React.FC<CardsComponentProps> = ({
        cards,
        setCards,
        loadingCards
    }) => {

        const [showAddCard, setShowAddCard] = useState(false);
        const [editingId, setEditingId] = useState<string | null>(null);

        const [form, setForm] = useState<CardForm>({
            holder: "",
            type: "",
            number: "",
            month: "",
            year: "",
            cvv: "",
            isDefault: false
        });

        // ================================
        // INPUT
        // ================================
        const handleChange = (
            e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement>
        ) => {
            const { name, value, type } = e.target;

            const finalValue =
                type === "checkbox"
                    ? (e.target as HTMLInputElement).checked
                    : value;

            setForm(prev => ({
                ...prev,
                [name]: finalValue
            }));
        };

        // ================================
        // REFRESH
        // ================================
        const refreshCards = async () => {
            const response = await auth.CUSTOMER_CARDS.LIST();
            const data = await response.json();
            setCards(data?.data || []);
        };

        // ================================
        // SAVE
        // ================================
        const saveCard = async () => {
            const payload = {
                holder_name: form.holder,
                card_type: form.type,
                card_number: form.number,
                expiry_month: form.month,
                expiry_year: form.year,
                cvv: form.cvv,
                is_default: form.isDefault
            };

            const res = editingId
                ? await auth.CUSTOMER_CARDS.UPDATE(editingId, payload)
                : await auth.CUSTOMER_CARDS.ADD(payload);

            if (res.ok) {
                await refreshCards();
                resetForm();
            } else {
                alert("Failed to save card");
            }
        };

        // ================================
        // EDIT
        // ================================
        const editCard = (card: CustomerCard) => {
            setEditingId(card.encrypted_id);
            setShowAddCard(true);

            setForm({
                holder: card.holder_name,
                type: card.card_type,
                number: "**** **** **** " + card.last4,
                month: card.expiry_month,
                year: card.expiry_year,
                cvv: "",
                isDefault: card.is_default
            });
        };

        // ================================
        // DELETE
        // ================================
        const deleteCard = async (id: string) => {
            const prev = [...cards];
            setCards(cards.filter(c => c.encrypted_id !== id));

            const res = await auth.CUSTOMER_CARDS.DELETE(id);

            if (!res.ok) {
                setCards(prev);
                alert("Failed to delete card");
            }
        };

        // ================================
        // DEFAULT
        // ================================
        const setDefaultCard = async (id: string) => {
            const prev = [...cards];

            setCards(cards.map(c => ({
                ...c,
                is_default: c.encrypted_id === id
            })));

            const res = await auth.CUSTOMER_CARDS.SET_DEFAULT(id);

            if (!res.ok) {
                setCards(prev);
                alert("Failed to set default");
            }
        };

        // ================================
        // RESET
        // ================================
        const resetForm = () => {
            setEditingId(null);
            setShowAddCard(false);
            setForm({
                holder: "",
                type: "",
                number: "",
                month: "",
                year: "",
                cvv: "",
                isDefault: false
            });
        };

        // ================================
        // UI
        // ================================
        if (loadingCards) {
            return (
                <div className="text-center py-4">
                    <div className="spinner-border text-primary" />
                </div>
            );
        }

        return (
            <div className="card dashboard-card">
                <div className="card-body">

                    {/* HEADER */}
                    <div className="d-flex justify-content-between mb-4">
                        <h4>
                            <i className="fas fa-credit-card me-2"></i>
                            Payment Cards
                        </h4>

                        {cards.length > 0 && (
                            <button
                                className="btn btn-sm"
                                style={{ backgroundColor: "#2f80ff", color: "white" }}
                                onClick={() => {
                                    resetForm();
                                    setShowAddCard(!showAddCard);
                                }}
                            >
                                <i className={`fas ${showAddCard ? "fa-times" : "fa-plus"} me-2`} />
                                {showAddCard ? "Cancel" : "Add Card"}
                            </button>
                        )}
                    </div>

                    {/* EMPTY STATE */}
                    {!showAddCard && cards.length === 0 && (
                        <div className="text-center py-5">
                            <i className="fas fa-credit-card fa-3x text-muted mb-3"></i>
                            <h5>No payment cards added</h5>
                            <p className="text-muted mb-3">
                                Add a card to make faster payments.
                            </p>
                            <button
                                className="btn"
                                style={{ backgroundColor: "#2f80ff", color: "white" }}
                                onClick={() => setShowAddCard(true)}
                            >
                                Add Your First Card
                            </button>
                        </div>
                    )}

                    {/* ADD / EDIT FORM */}
                    {showAddCard && (
                        <div className="card mb-4 border-primary">
                            <div className="card-body">
                                <h5 className="text-primary mb-3">
                                    {editingId ? "Update Card" : "Add New Card"}
                                </h5>

                                <input className="form-control mb-2" name="holder" placeholder="Card Holder Name" value={form.holder} onChange={handleChange} />
                                <select className="form-select mb-2" name="type" value={form.type} onChange={handleChange}>
                                    <option value="">--- Select Card Type ---</option>
                                    <option value="visa">Visa</option>
                                    <option value="master-card">Master Card</option>
                                    <option value="rupay">RuPay</option>
                                    <option value="american-express">American Express</option>
                                </select>

                                <input
                                    className="form-control mb-2"
                                    name="number"
                                    placeholder="Card Number"
                                    value={form.number}
                                    inputMode="numeric"
                                    pattern="[0-9 ]*"
                                    maxLength={19}
                                    onChange={(e) => {
                                        // remove non-digits
                                        const raw = e.target.value.replace(/\D/g, "");

                                        // group digits (4-4-4-4)
                                        const formatted = raw
                                            .slice(0, 16)
                                            .replace(/(\d{4})(?=\d)/g, "$1 ");

                                        setForm(prev => ({
                                            ...prev,
                                            number: formatted
                                        }));
                                    }}
                                />

                                <div className="row mb-2">
                                    <div className="col">
                                        <select
                                            className="form-select"
                                            name="month"
                                            value={form.month}
                                            onChange={handleChange}
                                        >
                                            <option value="">MM</option>
                                            {Array.from({ length: 12 }, (_, i) => {
                                                const month = String(i + 1).padStart(2, "0");
                                                return (
                                                    <option key={month} value={month}>
                                                        {month}
                                                    </option>
                                                );
                                            })}
                                        </select>
                                    </div>
                                    <div className="col">
                                        <select
                                            className="form-select"
                                            name="year"
                                            value={form.year}
                                            onChange={handleChange}
                                        >
                                            <option value="">YYYY</option>
                                            {Array.from({ length: 12 }, (_, i) => {
                                                const year = new Date().getFullYear() + i;
                                                return (
                                                    <option key={year} value={year}>
                                                        {year}
                                                    </option>
                                                );
                                            })}
                                        </select>
                                    </div>
                                    <div className="col">
                                        <input
                                            className="form-control"
                                            name="cvv"
                                            placeholder="CVV"
                                            value={form.cvv}
                                            maxLength={4}
                                            inputMode="numeric"
                                            pattern="[0-9]*"
                                            onChange={(e) => {
                                                const value = e.target.value.replace(/\D/g, "");
                                                setForm(prev => ({
                                                    ...prev,
                                                    cvv: value
                                                }));
                                            }}
                                        />
                                    </div>
                                </div>

                                <div className="form-check mb-3">
                                    <input className="form-check-input" type="checkbox" name="isDefault" id='checkDefault' checked={form.isDefault} onChange={handleChange} />
                                    <label htmlFor='checkDefault' className="form-check-label">Set as Default</label>
                                </div>

                                <button
                                    className="btn w-100"
                                    style={{ backgroundColor: "#2f80ff", color: "white" }}
                                    onClick={saveCard}
                                >
                                    {editingId ? "Update Card" : "Save Card"}
                                </button>
                            </div>
                        </div>
                    )}

                    {/* CARD LIST */}
                    {cards.length > 0 && (
                        <div className="row">
                            {cards.map(card => (
                                <div key={card.encrypted_id} className="col-lg-6">
                                    <div className="card mb-3 border-primary">
                                        <div className="card-body d-flex justify-content-between">
                                            <div>
                                                <h6>{card.card_type}</h6>
                                                <p className="mb-1">**** **** **** {card.last4}</p>
                                                <small>Expires: {card.expiry_month}/{card.expiry_year}</small>

                                                {card.is_default && (
                                                    <span className="badge bg-primary ms-2">Default</span>
                                                )}

                                                {!card.is_default && (
                                                    <button
                                                        className="btn btn-sm btn-outline-primary mt-2"
                                                        onClick={() => setDefaultCard(card.encrypted_id)}
                                                    >
                                                        Set as Default
                                                    </button>
                                                )}
                                            </div>

                                            <div>
                                                <button className="btn btn-sm btn-outline-secondary me-2" onClick={() => editCard(card)}>
                                                    <i className="fas fa-edit"></i>
                                                </button>
                                                <button className="btn btn-sm btn-outline-danger" onClick={() => deleteCard(card.encrypted_id)}>
                                                    <i className="fas fa-trash"></i>
                                                </button>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}

                </div>
            </div>
        );
    };


    // // Addresses Component
    // const AddressesComponent = () => (
    //     <div className="card dashboard-card">
    //         <div className="card-body">
    //             <div className="d-flex justify-content-between align-items-center mb-4">
    //                 <h4><i className="fas fa-map-marker-alt me-2"></i>Saved Addresses</h4>
    //                 <button
    //                     className="btn btn-sm"
    //                     style={{ backgroundColor: '#2f80ff', color: 'white' }}
    //                     onClick={() => setShowAddAddress(!showAddAddress)}
    //                 >
    //                     <i className={`fas ${showAddAddress ? 'fa-times' : 'fa-plus'} me-2`}></i>
    //                     {showAddAddress ? 'Cancel' : 'Add Address'}
    //                 </button>
    //             </div>

    //             {/* Add Address Form */}
    //             {showAddAddress && (
    //                 <div className="card mb-4" style={{ borderColor: '#2f80ff', borderWidth: '2px' }}>
    //                     <div className="card-body" style={{ backgroundColor: '#f8f9ff' }}>
    //                         <h5 className="mb-3" style={{ color: '#2f80ff' }}>
    //                             <i className="fas fa-plus-circle me-2"></i>Add New Address
    //                         </h5>
    //                         <div>
    //                             <div className="row mb-3">
    //                                 <div className="col-md-6">
    //                                     <label className="form-label">Full Name</label>
    //                                     <input
    //                                         type="text"
    //                                         className="form-control"
    //                                         placeholder="Enter full name"
    //                                     />
    //                                 </div>
    //                                 <div className="col-md-6">
    //                                     <label className="form-label">Phone Number</label>
    //                                     <input
    //                                         type="tel"
    //                                         className="form-control"
    //                                         placeholder="+91 9876543210"
    //                                     />
    //                                 </div>
    //                             </div>

    //                             <div className="mb-3">
    //                                 <label className="form-label">Address Line 1</label>
    //                                 <input
    //                                     type="text"
    //                                     className="form-control"
    //                                     placeholder="House No., Building Name"
    //                                 />
    //                             </div>

    //                             <div className="mb-3">
    //                                 <label className="form-label">Address Line 2</label>
    //                                 <input
    //                                     type="text"
    //                                     className="form-control"
    //                                     placeholder="Road name, Area, Colony"
    //                                 />
    //                             </div>

    //                             <div className="row mb-3">
    //                                 <div className="col-md-4">
    //                                     <label className="form-label">City</label>
    //                                     <input
    //                                         type="text"
    //                                         className="form-control"
    //                                         placeholder="City"
    //                                     />
    //                                 </div>
    //                                 <div className="col-md-4">
    //                                     <label className="form-label">State</label>
    //                                     <select className="form-select">
    //                                         <option>Select State</option>
    //                                         <option>Maharashtra</option>
    //                                         <option>Delhi</option>
    //                                         <option>Karnataka</option>
    //                                         <option>Tamil Nadu</option>
    //                                         <option>Gujarat</option>
    //                                         <option>West Bengal</option>
    //                                         <option>Haryana</option>
    //                                     </select>
    //                                 </div>
    //                                 <div className="col-md-4">
    //                                     <label className="form-label">PIN Code</label>
    //                                     <input
    //                                         type="text"
    //                                         className="form-control"
    //                                         placeholder="400001"
    //                                         maxLength={6}
    //                                     />
    //                                 </div>
    //                             </div>

    //                             <div className="mb-3">
    //                                 <label className="form-label">Address Type</label>
    //                                 <div>
    //                                     <div className="form-check form-check-inline">
    //                                         <input className="form-check-input" type="radio" name="addressType" id="home" defaultChecked />
    //                                         <label className="form-check-label" htmlFor="home">
    //                                             <i className="fas fa-home me-1"></i>Home
    //                                         </label>
    //                                     </div>
    //                                     <div className="form-check form-check-inline">
    //                                         <input className="form-check-input" type="radio" name="addressType" id="office" />
    //                                         <label className="form-check-label" htmlFor="office">
    //                                             <i className="fas fa-building me-1"></i>Office
    //                                         </label>
    //                                     </div>
    //                                     <div className="form-check form-check-inline">
    //                                         <input className="form-check-input" type="radio" name="addressType" id="other" />
    //                                         <label className="form-check-label" htmlFor="other">
    //                                             <i className="fas fa-map-pin me-1"></i>Other
    //                                         </label>
    //                                     </div>
    //                                 </div>
    //                             </div>

    //                             <div className="form-check mb-3">
    //                                 <input className="form-check-input" type="checkbox" id="defaultAddress" />
    //                                 <label className="form-check-label" htmlFor="defaultAddress">
    //                                     Set as default delivery address
    //                                 </label>
    //                             </div>

    //                             <button
    //                                 type="button"
    //                                 className="btn w-100"
    //                                 style={{ backgroundColor: '#2f80ff', color: 'white' }}
    //                                 onClick={() => setShowAddAddress(false)}
    //                             >
    //                                 <i className="fas fa-save me-2"></i>Save Address
    //                             </button>
    //                         </div>
    //                     </div>
    //                 </div>
    //             )}

    //             {/* Existing Addresses */}
    //             {/* Existing Addresses */}
    //             <div className="row">
    //                 {addresses.map(addr => (
    //                     <div key={addr.id} className="col-lg-6">
    //                         <div className="card mb-3 h-100" style={{ borderColor: '#2f80ff' }}>
    //                             <div className="card-body">
    //                                 <div className="d-flex justify-content-between align-items-start">
    //                                     <div>
    //                                         <h6><i className="fas fa-home me-2" style={{ color: '#2f80ff' }}></i>{addr.type}</h6>
    //                                         <p className="mb-1">{addr.address}</p>
    //                                         {addr.isDefault && <span className="badge" style={{ backgroundColor: '#2f80ff' }}>Default Address</span>}
    //                                     </div>
    //                                     <div>
    //                                         <button className="btn btn-sm btn-outline-secondary me-2"><i className="fas fa-edit"></i></button>
    //                                         <button className="btn btn-sm btn-outline-danger"><i className="fas fa-trash"></i></button>
    //                                     </div>
    //                                 </div>
    //                             </div>
    //                         </div>
    //                     </div>
    //                 ))}
    //             </div>
    //         </div>
    //     </div>
    // );

    // Payment History Component
    const PaymentComponent = () => {
        const [payments, setPayments] = useState<any[]>([]);
        const [loading, setLoading] = useState<boolean>(true);
        const [error, setError] = useState<string | null>(null);

        useEffect(() => {
            let isMounted = true;

            async function fetchPayments() {
                try {
                    const response = await auth.CUSTOMER_PAYMENTS();

                    if (!response.ok) {
                        throw new Error(`HTTP error! Status: ${response.status}`);
                    }

                    const data = await response.json();

                    if (isMounted) {
                        setPayments(data?.data || []);
                    }
                } catch (err) {
                    console.error("Failed to fetch payments:", err);
                    if (isMounted) {
                        setError("Failed to load payment history.");
                    }
                } finally {
                    if (isMounted) {
                        setLoading(false);
                    }
                }
            }

            fetchPayments();

            return () => {
                isMounted = false;
            };
        }, []);

        const getStatusBadge = (status?: string) => {
            switch ((status || "").toLowerCase()) {
                case "success":
                case "confirmed":
                    return <span className="badge bg-success">Completed</span>;
                case "pending":
                    return <span className="badge bg-warning text-dark">Pending</span>;
                case "failed":
                    return <span className="badge bg-danger">Failed</span>;
                default:
                    return <span className="badge bg-secondary">Unknown</span>;
            }
        };

        const handleDownloadInvoice = async (invoiceNumber?: string) => {
            if (!invoiceNumber) {
                alert("Invoice number is missing.");
                return;
            }

            try {
                const response = await auth.CUSTOMER_PAYMENT_INVOICE(encodeURIComponent(invoiceNumber));

                if (!response.ok) {
                    if (response.status === 404) {
                        throw new Error("Invoice file not found.");
                    }

                    if (response.status === 401) {
                        throw new Error("You are not authorized to download this invoice.");
                    }

                    throw new Error("Failed to download invoice.");
                }

                const blob = await response.blob();

                if (!blob || blob.size === 0) {
                    throw new Error("Invoice file is empty or missing.");
                }

                const fileURL = window.URL.createObjectURL(blob);

                window.open(fileURL, "_blank", "noopener,noreferrer");
                // const newTab = window.open(fileURL, "_blank", "noopener,noreferrer");

                // if (!newTab) {
                //     // Popup blocked → fallback download
                //     const a = document.createElement("a");
                //     a.href = fileURL;
                //     a.download = `${invoiceNumber}.pdf`;
                //     document.body.appendChild(a);
                //     a.click();
                //     document.body.removeChild(a);
                // }

                // Cleanup
                setTimeout(() => {
                    window.URL.revokeObjectURL(fileURL);
                }, 5000);

            } catch (error: any) {
                console.error("Invoice download failed:", error);

                const message =
                    error?.message ||
                    "Invoice is currently unavailable. Please contact support.";

                alert(message);
            }
        };

        return (
            <div className="card dashboard-card">
                <div className="card-body">
                    <h4 className="mb-4">
                        <i
                            className="fas fa-receipt me-2"
                            style={{ color: "#2f80ff" }}
                        ></i>
                        Payment History
                    </h4>

                    {loading ? (
                        <div className="text-center py-4">
                            <div
                                className="spinner-border text-primary"
                                role="status"
                            >
                                <span className="visually-hidden">Loading...</span>
                            </div>
                        </div>
                    ) : error ? (
                        <div className="text-center py-4 text-danger">
                            {error}
                        </div>
                    ) : payments.length === 0 ? (
                        <div className="text-center py-4 text-muted">
                            No payment records found
                        </div>
                    ) : (
                        <div className="table-responsive">
                            <table className="table table-hover">
                                <thead style={{ backgroundColor: "#f8f9ff" }}>
                                    <tr>
                                        <th>Order ID</th>
                                        <th>Date</th>
                                        <th>Amount</th>
                                        <th>Status</th>
                                        <th>Action</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    {payments.map((payment) => (
                                        <tr key={payment.order_id || payment.order_number}>
                                            <td>
                                                <strong>
                                                    #{payment.order_number || "—"}
                                                </strong>
                                            </td>
                                            <td>
                                                {payment.created_at
                                                    ? new Date(
                                                        payment.created_at
                                                    ).toLocaleDateString("en-IN", {
                                                        day: "2-digit",
                                                        month: "short",
                                                        year: "numeric",
                                                    })
                                                    : "—"}
                                            </td>
                                            <td>
                                                <strong style={{ color: "#2f80ff" }}>
                                                    ₹
                                                    {Number(
                                                        payment.amount || 0
                                                    ).toLocaleString("en-IN")}
                                                </strong>
                                            </td>
                                            <td>
                                                {getStatusBadge(
                                                    payment.transaction_status ||
                                                    payment.order_status
                                                )}
                                            </td>
                                            <td>
                                                <button
                                                    className="btn btn-sm"
                                                    style={{
                                                        borderColor: "#2f80ff",
                                                        color: "#2f80ff",
                                                    }}
                                                    onClick={() =>
                                                        handleDownloadInvoice(
                                                            payment.invoice_number
                                                        )
                                                    }
                                                    disabled={!payment.invoice_number}
                                                    title={
                                                        payment.invoice_number
                                                            ? "Download Invoice"
                                                            : "Invoice not available"
                                                    }
                                                >
                                                    <i className="fas fa-download"></i>
                                                </button>
                                            </td>
                                        </tr>
                                    ))}
                                </tbody>
                            </table>
                        </div>
                    )}
                </div>
            </div>
        );
    };

    // Orders Component
    const [orders, setOrders] = useState<Order[]>([]);
    const [loadingOrders, setLoadingOrders] = useState(true);

    useEffect(() => {
        const fetchOrders = async () => {
            try {
                const response = await auth.CUSTOMER_ORDERS();

                if (!response.ok) {
                    throw new Error(`HTTP error! Status: ${response.status}`);
                }

                const data = await response.json();

                setOrders(data?.data || []);
            } catch (err) {
                console.error("Failed to fetch orders:", err);
            } finally {
                setLoadingOrders(false);
            }
        };

        fetchOrders();
    }, []);

    const OrdersComponent: React.FC<OrdersComponentProps> = ({
        orders,
        setOrders,
        loadingOrders
    }) => {

        if (loadingOrders) {
            return (
                <div className="text-center py-4">
                    <div className="spinner-border text-primary" role="status">
                        <span className="visually-hidden">Loading...</span>
                    </div>
                </div>
            );
        }

        return (
            <div className="card dashboard-card">
                <div className="card-body">
                    <h4 className="mb-4">
                        <i className="fas fa-box me-2" style={{ color: "#2f80ff" }}></i>
                        Order Status
                    </h4>

                    {orders.length === 0 ? (
                        <p>No orders available.</p>
                    ) : (
                        <div className="row">
                            {orders.map((order) => (
                                <div key={order.order_number} className="col-lg-6 mb-3">
                                    <div
                                        className="card mb-3 h-100"
                                        style={{ borderColor: "#2f80ff" }}
                                    >
                                        <div className="card-body">
                                            <div className="row align-items-center">
                                                <div className="col-md-4">
                                                    <h6 className="mb-0 text-truncate">
                                                        {order.order_number}
                                                    </h6>
                                                    <small className="text-muted">
                                                        {new Date(order.created_at).toLocaleDateString("en-GB")}
                                                    </small>
                                                </div>

                                                <div className="col-md-4">
                                                    <p className="mb-0">
                                                        <strong>₹{order.total}</strong>
                                                    </p>
                                                </div>

                                                <div className="col-md-4">
                                                    <span
                                                        className={`badge ${order.order_status === "pending"
                                                            ? "bg-warning"
                                                            : order.order_status === "confirmed"
                                                                ? "bg-success"
                                                                : "bg-danger"
                                                            }`}
                                                    >
                                                        {order.order_status}
                                                    </span>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    )}
                </div>
            </div>
        );
    };

    // Order Status Component
    const OrderStatusComponent = () => (
        <div className="card dashboard-card">
            <div className="card-body">
                <h4 className="mb-4"><i className="fas fa-box me-2" style={{ color: '#2f80ff' }}></i>Order Status & Tracking</h4>

                {/* Order List */}
                {!selectedOrder && (
                    <div className="row">
                        {orders.map(order => (
                            <div key={order.id} className="col-lg-6 mb-3">
                                <div className="card mb-3 h-100" style={{ borderColor: '#2f80ff' }}>
                                    <div className="card-body">
                                        <div className="row align-items-center">
                                            <div className="col-md-3">
                                                <h6 className="mb-0 text-truncate">{order.id}</h6>
                                                <small className="text-muted">{order.date}</small>
                                            </div>
                                            <div className="col-md-3">
                                                <p className="mb-0"><strong>₹{order.total}</strong></p>
                                                <small className="text-muted">{order.items} items</small>
                                            </div>
                                            <div className="col-md-3">
                                                <span className={`badge bg-${order.statusColor}`}>{order.status}</span>
                                            </div>
                                            <div className="col-md-3 text-end">
                                                <button
                                                    className="btn btn-sm btn-primary w-100"
                                                    style={{ backgroundColor: '#2f80ff' }}
                                                    onClick={() => setSelectedOrder(order)}
                                                >
                                                    <i className="fas fa-truck me-1"></i>Track
                                                </button>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        ))}
                    </div>
                )}

                {/* Order Tracking Detail */}
                {selectedOrder && (
                    <div>
                        <button
                            className="btn btn-sm btn-link mb-3 p-0"
                            style={{ color: '#2f80ff' }}
                            onClick={() => setSelectedOrder(null)}
                        >
                            <i className="fas fa-arrow-left me-2"></i>Back to Orders
                        </button>

                        <div className="card" style={{ borderColor: '#2f80ff', borderWidth: '2px' }}>
                            <div className="card-body">
                                <div className="d-flex justify-content-between align-items-start mb-4">
                                    <div>
                                        <h5>{selectedOrder.id}</h5>
                                        <p className="text-muted mb-0">Order Date: {selectedOrder.date}</p>
                                        <p className="text-muted mb-0">Total Amount: ₹{selectedOrder.total}</p>
                                    </div>
                                    <span className={`badge bg-${selectedOrder.statusColor} fs-6`}>
                                        {selectedOrder.status}
                                    </span>
                                </div>

                                {/* Tracking Timeline */}
                                <div className="tracking-timeline">
                                    {selectedOrder.trackingSteps.map((step, index) => (
                                        <div key={index} className="tracking-step d-flex mb-4">
                                            <div className="me-3">
                                                <div
                                                    className={`rounded-circle d-flex align-items-center justify-content-center`}
                                                    style={{
                                                        width: '40px',
                                                        height: '40px',
                                                        backgroundColor: step.completed ? '#2f80ff' : '#6c757d'
                                                    }}
                                                >
                                                    <i className={`fas ${step.completed ? 'fa-check' : 'fa-circle'} text-white`}></i>
                                                </div>
                                                {index < selectedOrder.trackingSteps.length - 1 && (
                                                    <div
                                                        className="ms-3"
                                                        style={{
                                                            width: '3px',
                                                            height: '60px',
                                                            opacity: 0.5,
                                                            backgroundColor: step.completed ? '#2f80ff' : '#6c757d'
                                                        }}
                                                    ></div>
                                                )}
                                            </div>
                                            <div>
                                                <h6 style={{ color: step.completed ? '#2f80ff' : '#6c757d' }}>
                                                    {step.step}
                                                </h6>
                                                <p className="text-muted mb-0">{step.date}</p>
                                            </div>
                                        </div>
                                    ))}
                                </div>

                                {/* Additional Info */}
                                <div className="mt-4 p-3 rounded" style={{ backgroundColor: '#f8f9ff' }}>
                                    <h6><i className="fas fa-info-circle me-2" style={{ color: '#2f80ff' }}></i>Delivery Information</h6>
                                    <p className="mb-1"><strong>Expected Delivery:</strong> 30 Dec 2024</p>
                                    <p className="mb-0"><strong>Delivery Address:</strong> 123 Main St, Mumbai, MH 400001</p>
                                </div>
                            </div>
                        </div>
                    </div>
                )}
            </div>
        </div>
    );

    // Dashboard Overview Component
    const DashboardOverviewComponent = () => (
        <div className="container-fluid p-0">
            <h4 className="mb-4"><i className="fas fa-th-large me-2" style={{ color: '#2f80ff' }}></i>Dashboard Overview</h4>
            <div className="row g-4">
                {/* Payment Cards - col-lg-4 */}
                <div className="col-lg-4">
                    <div className="card dashboard-card h-100">
                        <div className="card-body">
                            <div className="d-flex justify-content-between align-items-center mb-3">
                                <h5 className="card-title mb-0"><i className="fas fa-credit-card me-2" style={{ color: '#2f80ff' }}></i>Cards</h5>
                                <button className="btn btn-sm btn-link" onClick={() => setActiveTab('cards')}>
                                    Manage
                                </button>
                            </div>
                            {cards.slice(0, 2).map(card => (
                                <div key={card.encrypted_id} className="d-flex justify-content-between align-items-center mb-2 p-2 rounded" style={{ backgroundColor: '#f8f9ff', border: '1px solid #e9ecef' }}>
                                    <div className="d-flex align-items-center">
                                        <i className={`fab fa-cc-${card.card_type.toLowerCase()} fa-lg me-3`} style={{ color: '#2f80ff' }}></i>
                                        <div>
                                            <p className="mb-0 fw-bold">**** {card.last4}</p>
                                            <small className="text-muted">{card.expiry_month}/{card.expiry_year}</small>
                                        </div>
                                    </div>
                                    {card.is_default && <span className="badge bg-primary rounded-pill">Default</span>}
                                </div>
                            ))}
                        </div>
                    </div>
                </div>

                {/* Saved Addresses - col-lg-4 */}
                {/* <div className="col-lg-4">
                    <div className="card dashboard-card h-100">
                        <div className="card-body">
                            <div className="d-flex justify-content-between align-items-center mb-3">
                                <h5 className="card-title mb-0"><i className="fas fa-map-marker-alt me-2" style={{ color: '#2f80ff' }}></i>Addresses</h5>
                                <button className="btn btn-sm btn-link" onClick={() => setActiveTab('addresses')}>
                                    Manage
                                </button>
                            </div>
                            {addresses.slice(0, 2).map(addr => (
                                <div key={addr.id} className="mb-2 p-2 rounded" style={{ backgroundColor: '#f8f9ff', border: '1px solid #e9ecef' }}>
                                    <div className="d-flex justify-content-between">
                                        <strong>{addr.type}</strong>
                                        {addr.isDefault && <span className="badge bg-success">Default</span>}
                                    </div>
                                    <p className="mb-0 text-truncate text-muted small">{addr.address}</p>
                                </div>
                            ))}
                        </div>
                    </div>
                </div> */}

                {/* Order Status - col-lg-4 */}
                <div className="col-lg-4">
                    <div className="card dashboard-card h-100">
                        <div className="card-body">
                            <div className="d-flex justify-content-between align-items-center mb-3">
                                <h5 className="card-title mb-0"><i className="fas fa-box me-2" style={{ color: '#2f80ff' }}></i>Order Status</h5>
                                <button className="btn btn-sm btn-link" onClick={() => setActiveTab('orders')}>
                                    View All
                                </button>
                            </div>
                            {loadingOrders ? (
                                <div className="text-center py-5">
                                    <div className="spinner-border text-primary" />
                                </div>
                            ) : orders.length === 0 ? (
                                <div className="text-center py-4">
                                    <i
                                        className="fa-solid fa-cart-shopping fa-2x"
                                        style={{ color: "#2f80ff" }}
                                    ></i>
                                    <p className="mt-3 text-muted">No orders found.</p>
                                </div>
                            ) : (
                                orders.slice(0, 2).map((order) => {

                                    const statusColor =
                                        order.order_status === "pending"
                                            ? "warning"
                                            : order.order_status === "confirmed"
                                                ? "success"
                                                : "danger";

                                    const progress =
                                        order.order_status === "confirmed" ? "100%" : "60%";

                                    return (
                                        <div key={order.order_number} className="mb-3">
                                            <div className="d-flex justify-content-between mb-1">
                                                <span className="small fw-bold">
                                                    {order.order_number}
                                                </span>

                                                <span className={`badge bg-${statusColor} smaller`}>
                                                    {order.order_status}
                                                </span>
                                            </div>

                                            <div className="progress" style={{ height: "6px" }}>
                                                <div
                                                    className={`progress-bar bg-${statusColor}`}
                                                    role="progressbar"
                                                    style={{ width: progress }}
                                                ></div>
                                            </div>

                                            <small className="text-muted d-block mt-1">
                                                {new Date(order.created_at).toLocaleDateString("en-GB")}
                                            </small>
                                        </div>
                                    );
                                })
                            )}
                        </div>
                    </div>
                </div>

                {/* Shopping Cart - col-lg-6 */}
                <div className="col-lg-6">
                    <div className="card dashboard-card h-100">
                        <div className="card-body">
                            <div className="d-flex justify-content-between align-items-center mb-3">
                                <h5 className="card-title mb-0">
                                    <i
                                        className="fas fa-shopping-cart me-2"
                                        style={{ color: "#2f80ff" }}
                                    ></i>
                                    Cart Summary
                                </h5>
                                <button
                                    className="btn btn-sm btn-link"
                                    onClick={() => setActiveTab("cart")}
                                >
                                    View Cart
                                </button>
                            </div>

                            {cartItems.length === 0 ? (
                                <div className="text-center py-4">
                                    <i
                                        className="fa-solid fa-cart-shopping fa-2x"
                                        style={{ color: "#2f80ff" }}
                                    ></i>
                                    <p className="mt-3 text-muted">No items in the cart.</p>
                                </div>
                            ) : (
                                <>
                                    <div
                                        className="d-flex align-items-center justify-content-between p-3 rounded mb-3"
                                        style={{ backgroundColor: "#f8f9ff" }}
                                    >
                                        <div>
                                            <h3 className="mb-0 fw-bold">{cartItems.length}</h3>
                                            <small className="text-muted">Items in Cart</small>
                                        </div>

                                        <div className="text-end">
                                            <h4 className="mb-0 text-primary">
                                                ₹
                                                {cartItems.reduce(
                                                    (sum, item) => sum + item.price * item.qty,
                                                    0
                                                )}
                                            </h4>
                                            <small className="text-muted">Total Value</small>
                                        </div>
                                    </div>

                                    {isLoggedIn && (
                                        <button
                                            className="btn w-100 btn-primary btn-sm"
                                            onClick={() => setActiveTab("cart")}
                                        >
                                            Proceed to Checkout
                                        </button>
                                    )}
                                </>
                            )}
                        </div>
                    </div>
                </div>

                {/* Profile Information - col-lg-6 (Small Box) */}
                <div className="col-lg-6">
                    <div className="card dashboard-card h-100">
                        <div className="card-body">
                            {loadingProfile ? (
                                <div className="text-center py-5">
                                    <div className="spinner-border text-primary" />
                                </div>
                            ) : userProfile ? (
                                <>
                                    <div className="d-flex justify-content-between align-items-center mb-3">
                                        <h5 className="card-title mb-0"><i className="fas fa-user me-2" style={{ color: '#2f80ff' }}></i>Profile</h5>
                                        <button className="btn btn-sm btn-link" onClick={() => setActiveTab('profile')}>
                                            Edit
                                        </button>
                                    </div>
                                    <div className="text-center p-3">
                                        <img
                                            src={userProfile.profile_photo!}
                                            className="rounded-circle mb-2"
                                            alt="Profile"
                                            style={{ border: '2px solid #2f80ff', width: 120, height: 120, objectFit: "cover" }}
                                        />
                                        <h6 className="mb-0">
                                            {[userProfile.first_name, userProfile.last_name].filter(Boolean).join(" ")}
                                        </h6>
                                        <small className="text-muted">{userProfile.email}</small>
                                        <div className="mt-3 d-flex justify-content-center gap-2">
                                            <span className="badge bg-light text-dark border">
                                                <i className="fas fa-phone me-1"></i>{userProfile.phone}
                                            </span>
                                        </div>
                                    </div>
                                </>
                            ) : (
                                <div className="text-center text-muted py-5">Profile data not available</div>
                            )}
                        </div>
                    </div>
                </div>

            </div>
        </div>
    );

    // Change Password Component
    const ChangePasswordComponent = () => {
        const [form, setForm] = useState<PasswordForm>({
            currentPassword: "",
            newPassword: "",
            confirmPassword: ""
        });

        const [errors, setErrors] = useState<PasswordErrors>({});
        const [success, setSuccess] = useState("");
        const [strength, setStrength] = useState("");
        const [loading, setLoading] = useState<boolean>(false);

        const [show, setShow] = useState({
            current: false,
            new: false,
            confirm: false
        });

        const handleChange = (e: React.ChangeEvent<HTMLInputElement>) => {
            setForm({ ...form, [e.target.name]: e.target.value });
            setErrors({});
            setSuccess("");
        };

        // Password strength checker
        const checkStrength = (password: string) => {
            let score = 0;
            if (password.length >= 8) score++;
            if (/[A-Z]/.test(password)) score++;
            if (/[a-z]/.test(password)) score++;
            if (/[0-9]/.test(password)) score++;
            if (/[^A-Za-z0-9]/.test(password)) score++;

            if (score <= 2) return "Weak";
            if (score === 3 || score === 4) return "Medium";
            return "Strong";
        };

        const validate = () => {
            const newErrors: PasswordErrors = {};

            if (!form.currentPassword)
                newErrors.currentPassword = "Current password is required";

            if (!form.newPassword)
                newErrors.newPassword = "New password is required";
            else if (form.newPassword.length < 8)
                newErrors.newPassword = "Password must be at least 8 characters";
            else if (!/[A-Z]/.test(form.newPassword))
                newErrors.newPassword = "Must include at least one uppercase letter";
            else if (!/[0-9]/.test(form.newPassword))
                newErrors.newPassword = "Must include at least one number";
            else if (!/[^A-Za-z0-9]/.test(form.newPassword))
                newErrors.newPassword = "Must include a special character";

            if (!form.confirmPassword)
                newErrors.confirmPassword = "Please confirm your password";
            else if (form.newPassword !== form.confirmPassword)
                newErrors.confirmPassword = "Passwords do not match";

            return newErrors;
        };

        const handleSubmit = async () => {
            const validationErrors = validate();

            if (Object.keys(validationErrors).length > 0) {
                setErrors(validationErrors);
                return;
            }

            const payload = {
                current_password: form.currentPassword,
                new_password: form.newPassword,
                new_password_confirmation: form.confirmPassword
            };

            try {
                setLoading(true);
                setErrors({});
                setSuccess("");

                const res = await auth.CUSTOMER_UPDATE_PASSWORD(payload);

                let data: any;

                // fetch response
                if (res instanceof Response) {
                    const json = await res.json();

                    if (!res.ok) {
                        // Laravel validation errors
                        if (json.errors) {
                            setErrors({
                                currentPassword: json.errors.current_password?.[0],
                                newPassword: json.errors.new_password?.[0],
                                confirmPassword:
                                    json.errors.new_password_confirmation?.[0],
                            });
                            return;
                        }

                        throw new Error(json.message || "Password update failed");
                    }

                    data = json;
                } else {
                    // axios response
                    data = res;

                    if (data.status === false) {
                        throw new Error(data.message || "Password update failed");
                    }
                }

                // ✅ success
                setSuccess(data.message || "Password updated successfully!");

                setForm({
                    currentPassword: "",
                    newPassword: "",
                    confirmPassword: ""
                });

                setStrength("");

                setTimeout(() => {
                    auth.LOGOUT();
                    router.replace("/login");
                }, 1500);

            } catch (error: any) {
                setSuccess("");

                setErrors({
                    currentPassword: error.message
                });
            } finally {
                setLoading(false);
            }
        };

        return (
            <div className="card dashboard-card">
                <div className="card-body">
                    <h4 className="mb-4">
                        <i className="fas fa-key me-2" style={{ color: "#2f80ff" }}></i>
                        Change Password
                    </h4>

                    <div className="row justify-content-center">
                        <div className="col-md-8">

                            <div className="alert" style={{ backgroundColor: '#f8f9ff', borderColor: '#2f80ff', color: '#000' }}>
                                <i className="fas fa-shield-alt me-2" style={{ color: '#2f80ff' }}></i>
                                Make sure your password is strong and secure. Use at least 8 characters with a mix of letters, numbers, and symbols.
                            </div>

                            {success && (
                                <div className="alert alert-success">
                                    <i className="fas fa-check-circle me-2"></i>
                                    {success}
                                </div>
                            )}

                            <div className="mb-3">
                                <label className="form-label">
                                    <i className="fas fa-lock me-2"></i>Current Password
                                </label>
                                <div className="input-group">
                                    <input
                                        type={show.current ? "text" : "password"}
                                        name="currentPassword"
                                        className="form-control"
                                        value={form.currentPassword}
                                        onChange={handleChange}
                                    />
                                    <button
                                        type="button"
                                        className="btn btn-outline-secondary"
                                        onClick={() =>
                                            setShow({ ...show, current: !show.current })
                                        }
                                    >
                                        <i className={`fas ${show.current ? "fa-eye-slash" : "fa-eye"}`}></i>
                                    </button>
                                </div>
                                {errors.currentPassword && (
                                    <small className="text-danger">{errors.currentPassword}</small>
                                )}
                            </div>

                            <div className="mb-3">
                                <label className="form-label">
                                    <i className="fas fa-lock me-2"></i>New Password
                                </label>
                                <div className="input-group">
                                    <input
                                        type={show.new ? "text" : "password"}
                                        name="newPassword"
                                        className="form-control"
                                        value={form.newPassword}
                                        onChange={(e) => {
                                            handleChange(e);
                                            setStrength(checkStrength(e.target.value));
                                        }}
                                    />
                                    <button
                                        type="button"
                                        className="btn btn-outline-secondary"
                                        onClick={() => setShow({ ...show, new: !show.new })}
                                    >
                                        <i className={`fas ${show.new ? "fa-eye-slash" : "fa-eye"}`}></i>
                                    </button>
                                </div>

                                {strength && (
                                    <small
                                        className={
                                            strength === "Strong"
                                                ? "text-success"
                                                : strength === "Medium"
                                                    ? "text-warning"
                                                    : "text-danger"
                                        }
                                    >
                                        Strength: {strength}
                                    </small>
                                )}

                                {errors.newPassword && (
                                    <div className="text-danger">{errors.newPassword}</div>
                                )}
                            </div>

                            <div className="mb-4">
                                <label className="form-label">Confirm Password</label>
                                <div className="input-group">
                                    <input
                                        type={show.confirm ? "text" : "password"}
                                        name="confirmPassword"
                                        className="form-control"
                                        value={form.confirmPassword}
                                        onChange={handleChange}
                                    />
                                    <button
                                        type="button"
                                        className="btn btn-outline-secondary"
                                        onClick={() =>
                                            setShow({ ...show, confirm: !show.confirm })
                                        }
                                    >
                                        <i className={`fas ${show.confirm ? "fa-eye-slash" : "fa-eye"}`}></i>
                                    </button>
                                </div>
                                {errors.confirmPassword && (
                                    <small className="text-danger">{errors.confirmPassword}</small>
                                )}
                            </div>

                            <button
                                className="btn btn-lg w-100"
                                style={{ backgroundColor: "#2f80ff", color: "white" }}
                                onClick={handleSubmit}
                                disabled={loading}
                            >
                                {loading ? (
                                    <>
                                        <span className="spinner-border spinner-border-sm me-2"></span>
                                        Updating...
                                    </>
                                ) : (
                                    <>
                                        <i className="fas fa-check me-2"></i>
                                        Update Password
                                    </>
                                )}
                            </button>

                            <div className="mt-4 p-3 rounded" style={{ backgroundColor: "#f8f9ff" }}>
                                <h6>
                                    <i className="fas fa-lightbulb me-2" style={{ color: "#2f80ff" }}></i>
                                    Password Tips
                                </h6>
                                <ul className="mb-0">
                                    <li>Minimum 8 characters</li>
                                    <li>Uppercase & lowercase letters</li>
                                    <li>At least one number</li>
                                    <li>At least one special character</li>
                                </ul>
                            </div>

                        </div>
                    </div>
                </div>
            </div>
        );
    };

    // async function logout() {
    //     try {
    //         // Call backend + clear token (AUTH already does this)
    //         if (auth.LOGOUT) {
    //             await auth.LOGOUT();
    //         }

    //         // Redirect
    //         router.replace("/login");
    //     } catch (err) {
    //         console.error("Logout failed", err);
    //     }
    // }

    if (!isAuthenticated) {
        return (
            <div className="container-fluid" style={{ minHeight: '100vh', background: 'linear-gradient(135deg, rgb(239 245 255) 0%, rgb(152 192 255) 100%)' }}>
                <div className="container">
                    <div className="row justify-content-center align-items-center h-100">
                        <div className="col-md-5">
                            <AuthComponent />
                        </div>
                    </div>
                </div>
            </div>
        );
    }

    // 👇 render guard — SAFE, no hook order change
    if (checking) {
        return <div style={{ padding: 20 }}>Loading...</div>;
    }

    return (
        <div className="container-fluid p-0">

            <div className="d-flex" style={{ minHeight: '100vh', overflowX: 'hidden' }}>
                {/* Sidebar */}
                <div
                    className="dashboard-sidebar text-white flex-shrink-0"
                    style={{
                        width: isSidebarOpen ? '260px' : '70px',
                        backgroundColor: '#000000',
                        transition: 'width 0.3s ease',
                        overflow: 'hidden',
                        minHeight: '100vh',
                        position: 'relative'
                    }}
                >
                    <div className="p-0">
                        <div className={`d-flex align-items-center px-3 mt-3 mb-4 ${isSidebarOpen ? 'justify-content-between' : 'justify-content-center'}`} style={{ height: '40px' }}>
                            <h4 className={`mb-0 text-nowrap ${!isSidebarOpen ? 'd-none' : ''}`} style={{ overflow: 'hidden' }}>
                                <i className="fas fa-dashboard me-2" style={{ color: '#2f80ff' }}></i>
                                Dashboard
                            </h4>
                            <button
                                className="btn btn-sm btn-link text-white p-0"
                                onClick={() => setIsSidebarOpen(!isSidebarOpen)}
                                style={{ fontSize: '1.2rem', textDecoration: 'none' }}
                            >
                                <i className={`fas ${isSidebarOpen ? 'fa-chevron-left' : 'fa-bars'}`}></i>
                            </button>
                        </div>
                        <ul className="nav flex-column">
                            {[
                                { id: 'dashboard', icon: 'fa-th-large', label: 'Dashboard' },
                                { id: 'profile', icon: 'fa-user', label: 'Profile' },
                                { id: 'cart', icon: 'fa-shopping-cart', label: 'Cart' },
                                { id: 'cards', icon: 'fa-credit-card', label: 'Cards' },
                                // { id: 'addresses', icon: 'fa-map-marker-alt', label: 'Addresses' },
                                { id: 'orders', icon: 'fa-box', label: 'Orders' },
                                { id: 'payment', icon: 'fa-receipt', label: 'Payments' },
                                { id: 'change-password', icon: 'fa-key', label: 'Change Password' }
                            ].map(item => (
                                <li className="nav-item mb-2" key={item.id}>
                                    <a
                                        href="#"
                                        className={`nav-link text-white ${activeTab === item.id ? '' : ''} ${!isSidebarOpen ? 'text-center px-0' : ''}`}
                                        style={activeTab === item.id ? { backgroundColor: '#2f80ff' } : {}}
                                        onClick={(e) => { e.preventDefault(); setActiveTab(item.id); }}
                                        title={!isSidebarOpen ? item.label : ''}
                                    >
                                        <i className={`fas ${item.icon} ${isSidebarOpen ? 'me-2' : ''}`}></i>
                                        <span className={!isSidebarOpen ? 'd-none' : ''}>{item.label}</span>
                                    </a>
                                </li>
                            ))}

                            {/* <li className="nav-item mb-2">
                                <a
                                    href="#"
                                    className={`nav-link text-white ${!isSidebarOpen ? 'text-center px-0' : ''}`}
                                    onClick={(e) => { e.preventDefault(); logout(); return false; }}
                                    title={!isSidebarOpen ? 'Logout' : ''}
                                >
                                    <i className={`fas fa-sign-out-alt ${isSidebarOpen ? 'me-2' : ''}`}></i>
                                    <span className={!isSidebarOpen ? 'd-none' : ''}>Logout</span>
                                </a>
                            </li> */}
                        </ul>
                    </div>
                </div>

                {/* Main Content */}
                <div className="flex-grow-1 p-lg-4 p-0 p-sm-1" style={{ backgroundColor: '#f8f9fa', minWidth: 0 }}>

                    {activeTab === 'dashboard' && <DashboardOverviewComponent />}
                    {activeTab === 'profile' && (
                        <ProfileComponent
                            userProfile={userProfile}
                            setUserProfile={setUserProfile}
                            loadingProfile={loadingProfile}
                        />
                    )}
                    {activeTab === 'cart' && <CartComponent />}
                    {activeTab === 'payment' && <PaymentComponent />}
                    {activeTab === 'cards' && (
                        <CardsComponent
                            cards={cards}
                            setCards={setCards}
                            loadingCards={loadingCards}
                        />
                    )}
                    {/* {activeTab === 'addresses' && <AddressesComponent />} */}
                    {activeTab === 'orders' && (
                        <OrdersComponent
                            orders={orders}
                            setOrders={setOrders}
                            loadingOrders={loadingOrders}
                        />
                    )}
                    {activeTab === 'change-password' && <ChangePasswordComponent />}
                </div>
            </div>
        </div>
    );
}
