import { API_ENDPOINTS } from "@/lib/apiEndpoints";
import { Metadata } from "next";
import FreeDownloadClient from "./FreeDownloadClient";
import { buildPageMeta } from "@/lib/seo/generalMeta";

export const dynamic = "force-dynamic";

async function fetchFreeDownloadData() {
    try {
        const [res, res2, free_downloads_res] = await Promise.all([
            API_ENDPOINTS.PAGE(12),
            API_ENDPOINTS.SECTION(94),
            API_ENDPOINTS.FREE_DOWNLOADS(),
        ]);

        if (!res.ok) throw new Error(`HTTP ${res.status}`);
        if (!res2.ok) throw new Error(`HTTP ${res2.status}`);
        if (!free_downloads_res.ok) throw new Error(`HTTP ${free_downloads_res.status}`);

        const [pageData, sectionData, freeDownloadsData] = await Promise.all([
            res.json(),
            res2.json(),
            free_downloads_res.json(),
        ]);

        return {
            pageData,
            sectionData,
            freeDownloads: freeDownloadsData || { top_banner_files: [], files: [] },
        };
    } catch (err: any) {
        console.error("❌ Failed to load free download data", err);
        return null;
    }
}

export async function generateMetadata(): Promise<Metadata> {
    const data = await fetchFreeDownloadData();
    if (!data || !data.pageData) return { title: "Free Downloads" };
    return buildPageMeta(data.pageData, "/free-download", "Free Downloads");
}

export default async function FreeDownloadPage() {
    const data = await fetchFreeDownloadData();

    if (!data) {
        return <div className="text-danger text-center py-5">Failed to load free downloads.</div>;
    }

    return (
        <FreeDownloadClient
            pageData={data.pageData}
            sectionData={data.sectionData}
            freeDownloads={data.freeDownloads}
        />
    );
}
