"use client";

import React, { useEffect, useState } from "react";
import { AUTH, getToken } from "@/lib/apiEndpoints";
import InnerBanner from "../../components/InnerBanner";
import "../auth.css";
import { Mail, Lock, Eye, EyeOff } from "lucide-react";
import { useRouter, useSearchParams } from "next/navigation";

function PageLoader() {
    return (
        <div className="min-vh-100 d-flex align-items-center justify-content-center">
            <div className="spinner-border text-primary" />
        </div>
    );
}

type ErrorType = string | Record<string, string> | null;

const BUY_NOW_ITEM_KEY = "buy_now_item";

export default function LoginClient() {
    const router = useRouter();
    const searchParams = useSearchParams();
    const auth = AUTH();

    const redirect = searchParams.get("redirect");
    const intent = searchParams.get("intent");

    const [checking, setChecking] = useState(true);

    useEffect(() => {
        const token = getToken();

        if (token) {
            const hasBuyNow = !!sessionStorage.getItem(BUY_NOW_ITEM_KEY);

            // 🔥 Already logged in
            if (intent === "buy_now" && hasBuyNow) {
                router.replace("/checkout");
            } else if (intent === "buy_now" && redirect) {
                router.replace(redirect);
            } else {
                router.replace("/dashboard");
            }
        } else {
            setChecking(false);
        }
    }, [router, redirect, intent]);

    const [email, setEmail] = useState("");
    const [password, setPassword] = useState("");
    const [error, setError] = useState<ErrorType>(null);
    const [loading, setLoading] = useState(false);
    const [showPassword, setShowPassword] = useState(false);
    const [rememberMe, setRememberMe] = useState(false);

    function validate() {
        if (!email || !/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email))
            return "Please enter a valid email.";
        if (!password) return "Password is required.";
        if (!password || password.length < 6)
            return "Password must be at least 6 characters.";
        return null;
    }

    async function handleSubmit(e: React.FormEvent) {
        e.preventDefault();
        setError(null);

        const validationError = validate();
        if (validationError) return setError({ general: validationError });

        if (!("LOGIN" in auth) || !auth.LOGIN) {
            return router.replace("/dashboard");
        }

        setLoading(true);

        try {
            const res = await auth.LOGIN({
                email,
                password,
                remember: rememberMe,
            });

            // ⭐ treat token as success
            if (res?.access_token) {
                window.dispatchEvent(new Event("auth-login"));

                const hasBuyNow = !!sessionStorage.getItem(BUY_NOW_ITEM_KEY);

                // 🔥 Buy Now always goes to checkout
                if (intent === "buy_now" && hasBuyNow) {
                    router.replace("/checkout");
                } else if (intent === "buy_now" && redirect) {
                    router.replace(redirect);
                } else {
                    router.replace("/dashboard");
                }

                return;
            }

            if (res.errors) {
                setError(res.errors || {});
            } else {
                setError({ general: res.message });
            }
        } catch (err) {
            setError("Something went wrong. Please try again.");
        } finally {
            setLoading(false);
        }
    }

    if (checking) {
        return <PageLoader />;
    }

    return (
        <>
            <InnerBanner title="Sign In" breadcrumb="Sign In" />

            <div className="auth-page-wrapper">
                <div className="auth-card">
                    <div className="auth-card-header">
                        <h2 className="auth-title">Login</h2>
                    </div>

                    <form onSubmit={handleSubmit} autoComplete="off">
                        {error && (
                            <div
                                className="auth-alert"
                                style={{ marginBottom: "20px", color: "red" }}
                            >
                                {typeof error === "string"
                                    ? error
                                    : error.general ?? Object.values(error)[0]}
                            </div>
                        )}

                        {/* Email */}
                        <div className="auth-form-group">
                            <div className="auth-input-wrapper">
                                <Mail className="auth-input-icon" size={20} />
                                <input
                                    className="auth-input"
                                    type="email"
                                    placeholder="Enter your E-Mail"
                                    value={email}
                                    autoComplete="off"
                                    onChange={(e) => setEmail(e.target.value)}
                                />
                            </div>
                        </div>

                        {/* Password */}
                        <div className="auth-form-group">
                            <div className="auth-input-wrapper">
                                <Lock className="auth-input-icon" size={20} />
                                <input
                                    className="auth-input"
                                    type={showPassword ? "text" : "password"}
                                    placeholder="Enter your Password"
                                    value={password}
                                    autoComplete="new-password"
                                    onChange={(e) => setPassword(e.target.value)}
                                />
                                <button
                                    type="button"
                                    className="auth-password-toggle"
                                    onClick={() => setShowPassword(!showPassword)}
                                >
                                    {showPassword ? <EyeOff size={18} /> : <Eye size={18} />}
                                </button>
                            </div>
                        </div>

                        {/* Options */}
                        <div className="auth-options">
                            <label className="auth-checkbox-label">
                                <input
                                    type="checkbox"
                                    className="auth-checkbox"
                                    checked={rememberMe}
                                    onChange={(e) => setRememberMe(e.target.checked)}
                                />
                                <span>Remember me</span>
                            </label>
                            <a href="#" className="auth-link">
                                Forgot password?
                            </a>
                        </div>

                        {/* Submit Button */}
                        <button
                            type="submit"
                            className="btn-auth"
                            disabled={loading}
                        >
                            {loading ? "Logging in..." : "Login Now"}
                        </button>

                        {/* Footer Link */}
                        <div className="auth-footer-link">
                            Don't have an account?{" "}
                            <a href="/register" className="auth-link">
                                Signup now
                            </a>
                        </div>
                    </form>
                </div>
            </div>
        </>
    );
}
