"use client";

import React from "react";
import { Swiper, SwiperSlide } from "swiper/react";
import "swiper/css";
import Link from "next/link";
import { motion, type Variants } from "framer-motion";
import "./BlogsReviewsSection.css";

const fadeUp: Variants = {
    hidden: { opacity: 0, y: 30 },
    visible: {
        opacity: 1,
        y: 0,
        transition: { duration: 0.6, ease: "easeOut" },
    },
};

const cardFadeUp: Variants = {
    hidden: { opacity: 0, y: 20 },
    visible: (i: number) => ({
        opacity: 1,
        y: 0,
        transition: { delay: i * 0.15, duration: 0.5, ease: "easeOut" },
    }),
};

interface BlogsReviewsData {
    headline: string;
    description: string;
    blogs: any[];
}

interface BlogsReviewsSectionProps {
    data: BlogsReviewsData | null;
}

export default function BlogsReviewsSection({ data }: BlogsReviewsSectionProps) {
    if (!data || !data.blogs?.length) {
        return null;
    }

    const { headline, description, blogs } = data;

    return (
        <section className="home-blog-section py-5 bg-gray-200">
            <div className="container">
                <motion.div
                    className="text-center"
                    initial="hidden"
                    whileInView="visible"
                    viewport={{ once: true }}
                    variants={fadeUp}
                >
                    <div className="mb-5">
                        <h2 className="brs-heading mb-2">{headline}</h2>
                        <div
                            className="brs-subtext mb-4"
                            dangerouslySetInnerHTML={{ __html: description }}
                        />
                    </div>
                </motion.div>

                <Swiper
                    spaceBetween={20}
                    slidesPerView={3}
                    loop={blogs.length > 3}
                    breakpoints={{
                        320: { slidesPerView: 1 },
                        768: { slidesPerView: 2 },
                        992: { slidesPerView: 3 },
                    }}
                >
                    {blogs.map((b, i) => (
                        <SwiperSlide key={b.id}>
                            <motion.div
                                className="blog-card h-100 shadow-sm rounded-4 overflow-hidden"
                                custom={i}
                                initial="hidden"
                                whileInView="visible"
                                viewport={{ once: true }}
                                variants={cardFadeUp}
                            >
                                <div className="blog-img-wrapper">
                                    <span className="category-badge">{b.category}</span>

                                    <img
                                        src={b.image}
                                        alt={b.title}
                                        className="blog-img"
                                        width="400"
                                        height="250"
                                        loading="lazy"
                                    />
                                </div>

                                <div className="blog-content">
                                    <p className="text-muted mb-2">
                                        <i className="fa-regular fa-calendar me-2"></i>
                                        {b.date}
                                    </p>
                                    <h2 className="blog-title">{b.title}</h2>
                                    <p className="blog-excerpt">{b.short_description}</p>
                                    <Link
                                        href={`/blog/${b.slug}`}
                                        className="read-more d-inline-flex align-items-center gap-2"
                                    >
                                        Read More <i className="fa-solid fa-arrow-right"></i>
                                    </Link>
                                </div>
                            </motion.div>
                        </SwiperSlide>
                    ))}
                </Swiper>
            </div>
        </section>
    );
}
