"use client";

import React from "react";
import { getToken } from "@/lib/apiEndpoints";
import Link from "next/link";
import { useCart } from "../context/CartContext";

export default function CartDrawer() {
    const isLoggedIn = !!getToken();
    const { items, isOpen, closeCart, changeQty, removeFromCart, cartTotal } = useCart();

    return (
        <>
            {/* Offcanvas Structure */}
            <div
                className={`offcanvas offcanvas-end ${isOpen ? "show" : ""}`}
                tabIndex={-1}
                id="cartOffcanvas"
                aria-labelledby="cartOffcanvasLabel"
                style={{ visibility: isOpen ? "visible" : "hidden" }}
            >
                <div className="offcanvas-header border-bottom">
                    <h5 className="offcanvas-title" id="cartOffcanvasLabel">Your Cart</h5>
                    <button
                        type="button"
                        className="btn-close text-reset"
                        onClick={closeCart}
                        aria-label="Close"
                    ></button>
                </div>

                <div className="offcanvas-body d-flex flex-column">
                    {items.length === 0 ? (
                        <div className="text-center my-auto">
                            <i className="fa-solid fa-cart-shopping fa-3x text-muted mb-3"></i>
                            <p className="text-muted">Your cart is empty</p>
                            <button className="btn btn-outline-primary" onClick={closeCart}>Start Shopping</button>
                        </div>
                    ) : (
                        <>
                            <div className="flex-grow-1 overflow-auto">
                                {items.map((item) => (
                                    <div key={item.id} className="d-flex align-items-center mb-4">
                                        <img
                                            src={item.img}
                                            alt={item.name}
                                            className="rounded"
                                            style={{ width: "70px", height: "70px", objectFit: "cover" }}
                                        />
                                        <div className="ms-3 flex-grow-1">
                                            <h6 className="mb-0 clamp-1">{item.name}</h6>
                                            <div className="text-muted small">₹{item.price}</div>
                                            <div className="d-flex align-items-center mt-2 justify-content-between">
                                                <div className="input-group input-group-sm" style={{ width: "90px" }}>
                                                    <button
                                                        className="btn btn-outline-secondary"
                                                        type="button"
                                                        onClick={() => changeQty(item.id, -1)}
                                                    >-</button>
                                                    <span className="input-group-text bg-white">{item.qty}</span>
                                                    <button
                                                        className="btn btn-outline-secondary"
                                                        type="button"
                                                        onClick={() => changeQty(item.id, 1)}
                                                    >+</button>
                                                </div>
                                                <button
                                                    className="btn btn-sm text-danger"
                                                    onClick={() => removeFromCart(item.id)}
                                                >
                                                    <i className="fa-solid fa-trash"></i>
                                                </button>
                                            </div>
                                        </div>
                                    </div>
                                ))}
                            </div>

                            <div className="border-top pt-3 mt-3">
                                <div className="d-flex justify-content-between mb-3">
                                    <span className="fw-bold">Total:</span>
                                    <span className="fw-bold fs-5">₹{cartTotal}</span>
                                </div>
                                <div className="d-grid gap-2">
                                    <Link href="/cart" className="btn btn-outline-dark" onClick={closeCart}>
                                        View Cart
                                    </Link>
                                    {isLoggedIn ? (
                                        <Link
                                            href="/checkout"
                                            className="btn btn-primary text-white"
                                            onClick={closeCart}
                                        >
                                            Checkout
                                        </Link>
                                    ) : (
                                        <Link
                                            href="/login"
                                            className="btn btn-primary text-white"
                                            onClick={closeCart}
                                        >
                                            Checkout
                                        </Link>
                                    )}
                                </div>
                            </div>
                        </>
                    )}
                </div>
            </div>

            {/* Backdrop */}
            {isOpen && (
                <div
                    className="offcanvas-backdrop fade show"
                    onClick={closeCart}
                    style={{ zIndex: 1040 }}
                ></div>
            )}
        </>
    );
}
