"use client";

import React from "react";
import Link from "next/link";
import { motion } from "framer-motion";
import type { Variants } from "framer-motion";
import "./PopularSection.css";

interface PopularItem {
    title: string;
    img: string;
    to: string;
}

interface PopularSectionData {
    headlineOne: string;
    descriptionOne: string;
    headlineTwo: string;
    titleTwo: string;
    descriptionTwo: string;
    items: PopularItem[];
}

interface PopularSectionProps {
    data: PopularSectionData | null;
}

const fadeUp: Variants = {
    hidden: { opacity: 0, y: 40 },
    show: { opacity: 1, y: 0, transition: { duration: 0.6, ease: "easeOut" } },
};

const staggerWrapper: Variants = {
    hidden: {},
    show: { transition: { staggerChildren: 0.12 } },
};

const cardAnim: Variants = {
    hidden: { opacity: 0, y: 35, scale: 0.96 },
    show: {
        opacity: 1,
        y: 0,
        scale: 1,
        transition: { duration: 0.45, ease: "easeOut" },
    },
};

const ctaFade: Variants = {
    hidden: { opacity: 0, y: 50 },
    show: { opacity: 1, y: 0, transition: { duration: 0.8 } },
};

export default function PopularSection({ data }: PopularSectionProps) {
    if (!data) return null;

    const {
        headlineOne,
        descriptionOne,
        headlineTwo,
        titleTwo,
        descriptionTwo,
        items,
    } = data;

    return (
        <section className="py-5" style={{ background: "#f7faff" }}>
            <motion.div
                className="container"
                initial="hidden"
                whileInView="show"
                viewport={{ once: true, amount: 0.2 }}
            >
                {/* Heading */}
                <motion.div className="text-center mb-4" variants={fadeUp}>
                    <h2
                        className="fw-bold"
                        dangerouslySetInnerHTML={{ __html: headlineOne }}
                    />

                    <div
                        className="text-muted mb-5 text-center"
                        dangerouslySetInnerHTML={{ __html: descriptionOne }}
                    />
                </motion.div>

                {/* Cards */}
                <motion.div className="row g-3 g-md-4" variants={staggerWrapper}>
                    {items.map((it) => (
                        <motion.div
                            key={it.title}
                            className="col-12 col-md-6 col-lg-3"
                            variants={cardAnim}
                        >
                            <div
                                className="popular-card d-flex align-items-center justify-content-between p-3 rounded-4 shadow-sm h-100 bg-white"
                                style={{ border: "1px solid rgba(0,0,0,0.06)" }}
                            >
                                <div className="d-flex align-items-center gap-3">
                                    <div
                                        className="rounded-3 d-flex align-items-center justify-content-center bg-primary bg-opacity-10"
                                        style={{ width: 60, height: 60 }}
                                    >
                                        <img
                                            src={it.img}
                                            alt={it.title}
                                            style={{
                                                width: "50px",
                                                height: "50px",
                                                objectFit: "contain",
                                                padding: "4px",
                                            }}
                                        />
                                    </div>

                                    <div
                                        className="popular-title fw-semibold small"
                                        style={{ lineHeight: 1.2 }}
                                    >
                                        {it.title}
                                    </div>
                                </div>

                                <Link
                                    href={it.to}
                                    className="popular-arrow d-inline-flex align-items-center justify-content-center rounded-circle bg-dark bg-opacity-10 text-dark"
                                    style={{ width: 30, height: 30 }}
                                >
                                    <i className="fa-solid fa-chevron-right small"></i>
                                </Link>
                            </div>
                        </motion.div>
                    ))}
                </motion.div>
            </motion.div>

            {/* CTA Block */}
            <motion.div
                className="mt-5"
                style={{
                    background: "linear-gradient(to right, #2f80ff, #6aa3ffff)",
                    color: "#fff",
                }}
                variants={ctaFade}
                initial="hidden"
                whileInView="show"
            >
                <div className="container py-5 text-center">
                    <h3
                        className="fw-bold mb-2"
                        dangerouslySetInnerHTML={{ __html: headlineTwo }}
                    />
                    <h4
                        className="fw-bold mb-3"
                        dangerouslySetInnerHTML={{ __html: titleTwo }}
                    />
                    <div
                        className="text-white-75 m-0 w-75 mx-auto"
                        dangerouslySetInnerHTML={{ __html: descriptionTwo }}
                    />
                </div>
            </motion.div>
        </section>
    );
}
