import { CartItem } from "@/context/CartContext";

const API_URL = process.env.NEXT_PUBLIC_API_URL!;

// ---- TOKEN (localStorage only) ----
export function getToken() {
    if (typeof window === "undefined") return null; // avoid SSR crash
    return localStorage.getItem("token");
}

function setToken(token: string) {
    if (typeof window === "undefined") return;
    localStorage.setItem("token", token);
    window.dispatchEvent(new Event("auth-changed"));
}

function clearToken() {
    if (typeof window === "undefined") return;
    localStorage.removeItem("token");
    window.dispatchEvent(new Event("auth-changed"));
}

export async function apiFetch(path: string, options: RequestInit = {}) {
    const token = getToken();

    const headers: HeadersInit = {
        ...(token ? { Authorization: `Bearer ${token}` } : {}),
        ...(options.headers as Record<string, string> | undefined),
    };

    // 🔥 Only set JSON header if NOT FormData
    if (!(options.body instanceof FormData)) {
        headers["Content-Type"] = "application/json";
    }

    return fetch(`${API_URL}${path}`, {
        method: options.method || "GET",
        credentials: "include",
        ...options,
        headers,
    });
}

type AuthClient = {
    LOGIN: (data: any) => Promise<any>;
    SIGNUP: (data: any) => Promise<any>;
    FORGOT_PASSWORD: (data: any) => Promise<any>;

    CUSTOMER_PROFILE: () => Promise<Response>;
    UPDATE_CUSTOMER_PROFILE: (data: any) => Promise<any>;
    CUSTOMER_CARDS: {
        LIST: () => Promise<Response>;
        ADD: (payload: any) => Promise<Response>;
        UPDATE: (cardId: string, payload: any) => Promise<Response>;
        DELETE: (cardId: string) => Promise<Response>;
        SET_DEFAULT: (cardId: string) => Promise<Response>;
    };
    CUSTOMER_PAYMENTS: () => Promise<Response>;
    CUSTOMER_PAYMENT_INVOICE: (invoice_number: any) => Promise<Response>;
    CUSTOMER_ORDERS: () => Promise<Response>;
    PLACE_ORDER: (data: any) => Promise<Response>;
    CUSTOMER_UPDATE_PASSWORD: (data: any) => Promise<Response>;
    LOGOUT: () => Promise<void>;
};

// ---- AUTH depending on login state ----
export function AUTH(): AuthClient {
    const token = getToken();

    return {
        LOGIN: async (data: any) => {
            const res = await apiFetch("login", {
                method: "POST",
                body: JSON.stringify(data),
            });

            const json = await res.json();

            if (json?.access_token) {
                setToken(json.access_token);
                window.dispatchEvent(new Event("auth-login"));
            }

            return json;
        },

        SIGNUP: async (data: any) => {
            const res = await apiFetch("signup", {
                method: "POST",
                body: JSON.stringify(data),
            });

            const json = await res.json();

            if (json?.access_token) {
                setToken(json.access_token);
                window.dispatchEvent(new Event("auth-login"));
            }

            return json;
        },

        FORGOT_PASSWORD: async (data: any) => {
            const res = await apiFetch("check-email-and-send-link", {
                method: "POST",
                body: JSON.stringify(data),
            });

            return res.json();
        },

        // ---------- PRIVATE ----------
        CUSTOMER_PROFILE: () => {
            if (!token) throw new Error("Not authenticated");
            return apiFetch("get-and-update-customer-profile");
        },

        UPDATE_CUSTOMER_PROFILE: async (data: FormData) => {
            const res = await apiFetch("get-and-update-customer-profile", {
                method: "POST",
                body: data,
            });

            return res.json();
        },

        CUSTOMER_CARDS: {
            LIST: () => apiFetch("get-customer-cards"),
            ADD: (payload: any) =>
                apiFetch("add-customer-card", {
                    method: "POST",
                    body: JSON.stringify(payload),
                }),
            UPDATE: (cardId: string, payload: any) =>
                apiFetch(`update-customer-card/${cardId}`, {
                    method: "PUT",
                    body: JSON.stringify(payload),
                }),
            DELETE: (cardId: string) =>
                apiFetch(`delete-customer-card/${cardId}`, {
                    method: "DELETE",
                }),
            SET_DEFAULT: (cardId: string) =>
                apiFetch(`set-default-card/${cardId}`, {
                    method: "PATCH",
                }),
        },

        CUSTOMER_PAYMENTS: () => {
            if (!token) throw new Error("Not authenticated");
            return apiFetch("get-customer-payments");
        },

        CUSTOMER_PAYMENT_INVOICE: (invoice_number: any) => {
            if (!token) throw new Error("Not authenticated");
            return apiFetch(`invoices/${invoice_number}`);
        },

        CUSTOMER_ORDERS: () => {
            if (!token) throw new Error("Not authenticated");
            return apiFetch("get-customer-orders");
        },

        PLACE_ORDER: (orderData: any) => {
            if (!token) throw new Error("Not authenticated");
            return apiFetch("place-order", {
                method: "POST",
                body: JSON.stringify(orderData),
            });
        },

        CUSTOMER_UPDATE_PASSWORD: (payload: any) => {
            if (!token) throw new Error("Not authenticated");
            return apiFetch("update-customer-password", {
                method: "POST",
                body: JSON.stringify(payload),
            });
        },

        LOGOUT: async () => {
            clearToken();
            window.dispatchEvent(new Event("auth-logout"));
        },
    };
}

export const API_ENDPOINTS = {
    SITE_STATUS: () => apiFetch("check-site-status"),

    COMPANY_CONTACT: () => apiFetch(`get-company-data`),
    // HEADER_LINKS: () => apiFetch(`get-header-links`),
    // FOOTER_LINKS: () => apiFetch(`get-footer-links`),
    BANNERS: () => apiFetch(`get-banners`),
    PAGE: (id: number) => apiFetch(`get-page?page_id=${id}`),
    SECTION: (id: number) => apiFetch(`get-page-section?section_id=${id}`),
    SUBSECTIONS: (page_id: number, id: number) => apiFetch(`get-page-subsections?page_id=${page_id}&section_id=${id}`),
    COVER_PAGE: () => apiFetch(`get-cover-pages`),
    FREE_DOWNLOADS: () => apiFetch(`get-free-downloads`),

    TESTIMONIALS: () => apiFetch(`get-testimonials`),
    FAQS: () => apiFetch(`get-faqs`),

    CONTACT_ENQUIRY_SAVE: (formData: FormData) => apiFetch(`contact-enquiry-data-save`, {
        method: "POST",
        body: formData,
    }),

    BLOGS: {
        LIST: () => apiFetch(`get-blogs`),
        DETAIL: (slug: string) => apiFetch(`get-blog/${slug}`),
    },

    ASSIGNMENTS: {
        CATEGORIES_COURSES: (type: string) => apiFetch(`get-assignment-categories-courses/${type}`),
        COURSE_DETAIL: (type: string, slug: string) => apiFetch(`get-course-assignments/${type}/${slug}`),
        DETAIL: (slug: string, code: string) => apiFetch(`get-single-course-assignment/${slug}/${code}`),
        CARDS: () => apiFetch(`get-assignment-cards`),
    },

    GUESS_PAPERS: {
        CATEGORIES_COURSES: () => apiFetch(`get-guess-paper-categories-courses`),
        COURSE_DETAIL: (slug: string) => apiFetch(`get-course-guess-papers/${slug}`),
        DETAIL: (slug: string, code: string) => apiFetch(`get-single-course-guess-paper/${slug}/${code}`),
    },

    PROJECTS: {
        CATEGORIES_COURSES: () => apiFetch(`get-project-categories-courses`),
        COURSE_DETAIL: (code: string) => apiFetch(`get-course-projects/${code}`),
        DETAIL: (code: string, projectId: string) => apiFetch(`get-single-course-project/${code}/${projectId}`),
        CARDS: () => apiFetch(`get-project-cards`),
    },

    HANDWRITTEN: {
        DATA: () => apiFetch(`get-handwritten-data`),
        SAVE: (formData: FormData) => apiFetch(`handwritten-data-save`, {
            method: "POST",
            body: formData,
        }),
    },

    CART: {
        LIST: () => apiFetch(`cart-user-list`),
        MERGE: (items: CartItem[]) => apiFetch("cart-merge", {
            method: "POST",
            body: JSON.stringify({ items }),
        }),
        ADD: (item: CartItem) => apiFetch(`cart-add`, {
            method: "POST",
            body: JSON.stringify(item),
        }),
        REMOVE: (item_id: string) => apiFetch(`cart-remove`, {
            method: "POST",
            body: JSON.stringify(item_id),
        }),
        UPDATE: (item_id: string, quantity: number) => apiFetch(`cart-update`, {
            method: "POST",
            body: JSON.stringify({ item_id, quantity }),
        }),
        CLEAR: () => apiFetch(`cart-clear`),
    },
} as const;
