import { Metadata } from "next";

function stripHtml(html: string): string {
    if (!html) return "";
    return html.replace(/<[^>]*>?/gm, "").trim();
}

/**
 * Metadata for a single Blog Post
 * e.g. /blog/some-post-slug
 */
export function buildBlogPostMeta(post: any, slug: string): Metadata {
    const description = stripHtml(
        post.meta_description ||
        post.excerpt ||
        post.short ||
        post.content?.substring(0, 160) || // Fallback to content if everything else fails
        ""
    );
    const title = post.meta_title || post.title || "Blog Post";
    const images = post.image ? [
        {
            url: post.image,
            width: 1200,
            height: 630,
            alt: title,
        },
    ] : undefined;

    return {
        title,
        description,
        keywords: post.meta_keywords,

        alternates: {
            canonical: `/blog/${slug}`,
        },

        openGraph: {
            title,
            description,
            url: `/blog/${slug}`,
            type: "article",
            images: images,
            publishedTime: post.date,
            authors: post.author ? [post.author] : undefined,
        },

        twitter: {
            card: "summary_large_image",
            title,
            description,
            images: images,
        },
    };
}
